--
--   Copyright (C) 2023 Ron Prusia
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program.  If not, see <http://www.gnu.org/licenses.
--
-- Identification
--
-- March, 2023
--         
-- Ron Prusia
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
entity fifo_out_ctrl is
generic (
  mod_id  : unsigned(15 downto 0) := x"1200";
  mod_len : unsigned(15 downto 0) := x"0102"
  );
port (
  clk       : in bit;						--Clock input 
  din       : in unsigned(15 downto 0);
  fifo_in   : in unsigned(15 downto 0);
--
  mem_rd    : out bit;
  dout      : out unsigned(15 downto 0)
  );
end fifo_out_ctrl;
--
architecture rtl of fifo_out_ctrl is
--
-- Constants
--
--  constant mod_id  : unsigned(15 downto 0) := x"1101"; --module identification
--  constant mod_len : unsigned(15 downto 0) := x"0202"; --length in words

--
-- Variables
--
  signal word_cnt      : unsigned(8 downto 0); --Count 
  signal word_cnt_en   : bit;
  signal word_cnt_sclr : bit;
  signal cmd_ff        : unsigned(15 downto 0); 
  signal cmd_ff_en     : bit;
  signal cmd_addr_ff   : unsigned(15 downto 0);
  signal cmd_addr_ff_en: bit;
  signal mod_addr_ff   : unsigned(15 downto 0);
  signal mod_addr_ff_en: bit;
  signal mod_ff        : unsigned(15 downto 0);
--
-- State machine
--
  type ss_type is (init,decode1,decode2,addr,read1,read2,write1,write2);
  signal ss_ps,ss_ns : ss_type;
--
begin
--
-- Clock processes, state machine and counter
--
  clk_proc : process(clk,ss_ns,cmd_ff,cmd_addr_ff,word_cnt,mod_addr_ff)
  begin
--  
    if (clk'event AND clk = '1') then 
      ss_ps <= ss_ns;
      mod_ff <= din; --hold input data one clock
--
      if (cmd_ff_en = '1') then cmd_ff <= mod_ff;--holds command, command length
      else cmd_ff <= cmd_ff;
      end if;
--      
      if (cmd_addr_ff_en = '1') then cmd_addr_ff <= mod_ff;--holds address
      else cmd_addr_ff <= cmd_addr_ff;
      end if;   
--
      if (word_cnt_sclr = '1') then word_cnt <= "000000000";--word counter for number of words
      elsif (word_cnt_en = '1') then word_cnt <= word_cnt + 1;
      else word_cnt <= word_cnt;
      end if;
--
      if (mod_addr_ff_en = '1') then mod_addr_ff <= mod_ff; --holds assigned module address, defined at power up
      else mod_addr_ff <= mod_addr_ff;
      end if;      
    end if;
--
  end process clk_proc;
--
-- Combinational process for state machine and output
--
  comb_proc : process (ss_ps,mod_ff,din,cmd_ff,cmd_addr_ff,mod_addr_ff,word_cnt,fifo_in)
  begin
--
  word_cnt_en <= '0';    --default values
  word_cnt_sclr <= '0';
  mod_addr_ff_en <= '0';
  cmd_ff_en <= '0';
  cmd_addr_ff_en <= '0';
  dout <= mod_ff;        --output defaults to din, delayed one clock
  mem_rd <= '0';
--
-- Initial sync word is defined as 0x0000 followed by 0xEB90
-- All logic is defined off the mod_ff input
--
  case ss_ps is 
    when init => --add one clock delay before setup
      word_cnt_sclr <= '1';
      if (mod_ff = x"0000") and (din = x"EB90") then ss_ns <= decode1;
      else ss_ns <= init;
      end if; 
    when decode1 => --wait one state for sync to be in mod_ff
      ss_ns <= decode2;
    when decode2 => --mod_ff equals command
      cmd_ff_en <= '1'; --latch command    
      if (mod_ff(15 downto 9) = "1111111") then ss_ns <= addr; --addr command
      elsif (mod_ff(15 downto 9) = "0000000") AND (mod_ff(8 downto 0) /= "000000000") then ss_ns <= read1; --read command
      elsif (mod_ff(15 downto 9) = "0000001") AND (mod_ff(8 downto 0) /= "000000000") then ss_ns <= write1;
      else ss_ns <= init; --all other commands are ignored.
      end if;
--
-- Address function
--    
    when addr => --mod_ff equals address of command
      mod_addr_ff_en <= '1'; --latch address given
      dout <= mod_ff + mod_len; --increment address 
      ss_ns <= init;
--
-- Read function
--    
    when read1 => --mod_ff equals address of command
      cmd_addr_ff_en <= '1'; --latch address given
      ss_ns <= read2;      
    when read2 => --increment through number of words, and output mux
      word_cnt_en <= '1';
      if (word_cnt = cmd_ff(8 downto 0)) then dout <= mod_ff;
      elsif (cmd_addr_ff + ("0000000"&word_cnt)) = mod_addr_ff   then dout <= mod_id;
      elsif (cmd_addr_ff + ("0000000"&word_cnt)) = mod_addr_ff+1 then dout <= mod_len;
      elsif ((cmd_addr_ff + ("0000000"&word_cnt)) > mod_addr_ff+1) AND
            ((cmd_addr_ff + ("0000000"&word_cnt)) < mod_addr_ff+mod_len) then
        dout <= fifo_in;
        mem_rd <= '1'; 
      end if;
      if (word_cnt = cmd_ff(8 downto 0)-1) then ss_ns <= init;
      else ss_ns <= read2;
      end if;
--
-- Write function, no writes for this module
--    
    when write1 => --mod_ff equals address of command
      cmd_addr_ff_en <= '1'; --latch address given
      ss_ns <= write2;      
    when write2 => --increment through number of words, and latch
      word_cnt_en <= '1';
      if (word_cnt = cmd_ff(8 downto 0)-1) then ss_ns <= init;
      else ss_ns <= write2;
      end if;
    when others => ss_ns <= init;
  end case;       
--    	    
  end process comb_proc;
end;
