--
--Outputs byte to serial port. Assumes din is held valid during the
--output serial stream. The byte sent line is asserted one clock at end of
--transmission of data.
--
--Ron Prusia
--Oct 2011
--May 2019 - Simplified
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
entity UART_tx is
port (
  clk	      : in bit;       --clock
  data_rdy    : in bit;       --Flag to indicate there is a byte to transmit
  data_in     : in unsigned(7 downto 0);  --Data to transmit
--
  tx          : out std_logic; --Output transmitted data
  byte_sent   : out bit       --Flag to indicate the byte was sent
  );
end UART_tx;
--
architecture rtl of UART_tx is
--
-- Constants
--
  constant baud_divide : unsigned(15 downto 0) := x"01B1"; --50Mhz, 230,400:00D8, 115,200:01B1
--
-- Variables
--
  signal din_ff         : unsigned(9 downto 0) := "1111111111"; --includes start stop bit
  signal data_load      : bit;
  signal data_shft      : bit; --Shift input data
  signal bit_cnt        : unsigned(3 downto 0);	--Output from mux to output register
  signal bit_cnt_sclr   : bit; --Sync clear
  signal bit_cnt_en     : bit; --Count enable
  signal baud_cnt       : unsigned(15 downto 0); --Baud rate counter
  signal baud_cnt_sclr  : bit; --Sync clear
  signal baud_cnt_cout  : bit; --Match of counter to baud rate input
--
  type ss_type is (init,bits_out);
  signal ss_ps,ss_ns : ss_type;
--
  begin
  tx <= din_ff(0);
--
-- Clock processes
--
  clk_proc : process(clk,ss_ns,baud_cnt_sclr,bit_cnt_sclr,data_rdy,data_in)
  begin
    if (clk'event AND clk = '1') then 
      ss_ps <= ss_ns;
--
      if (data_load = '1') then din_ff <= '1' & data_in & '0'; --latch input byte on data_in ready, lsb start bit
      elsif (data_shft = '1') then din_ff <= '1' & din_ff(9 downto 1); --shift right, shift in one
      else din_ff <= din_ff;
      end if;      
--
      if ((baud_cnt_sclr = '1') OR (baud_cnt = baud_divide)) then baud_cnt <= x"0000";
      else baud_cnt <= baud_cnt + 1;
      end if;
--
      if (bit_cnt_sclr = '1') then bit_cnt <= "0000";
      elsif (bit_cnt_en = '1') then bit_cnt <= bit_cnt + 1;
      else bit_cnt <= bit_cnt;
      end if;      
      
    end if;
  end process clk_proc;
--
-- The state machine waits for the data ready signal it then proceeds to output
-- the bits. Bit 0 is read first. The output is high, until output started. Start
-- stop bits are loaded with data byte.
--
  ss_comb_proc : process(ss_ps,ss_ns,data_rdy,bit_cnt,baud_cnt,din_ff)
  begin
--
    bit_cnt_sclr <= '0'; --default state for variables is not asserted
    bit_cnt_en <= '0';
    baud_cnt_sclr <= '0';
    byte_sent <= '0';
    data_shft <= '0';
    data_load <= '0';
--
    case ss_ps is
      when init => --Wait the ready signal
        bit_cnt_sclr <= '1';
        baud_cnt_sclr <= '1';
        if (data_rdy = '1') then 
          data_load <= '1';
          ss_ns <= bits_out;
        else ss_ns <= init;
        end if;
      when bits_out =>
        if ((baud_cnt = baud_divide) AND (bit_cnt /= "1001")) then 
           ss_ns <= bits_out;
           bit_cnt_en <= '1'; --Increment bit count, baud counter cleared in counter process
           data_shft <= '1'; -- Shift output data
        elsif ((baud_cnt = baud_divide) AND (bit_cnt = "1001")) then --don't shift data out to hold stop bit
          ss_ns <= init;
          byte_sent <= '1';
        else ss_ns <= bits_out;
        end if;
      when others => ss_ns <= init;
	end case;
  end process ss_comb_proc;
end;
