--
--  Copyright (C) 2020 Ron Prusia
--
--  This program is free software: you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation, either version 3 of the License, or
--  (at your option) any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--
--  You should have received a copy of the GNU General Public License
--  along with this program.  If not, see <http://www.gnu.org/licenses.
--
-- Receives UART data. 
-- Assume input data has 8 data bits, no parity
-- Framing error is not detected
--
-- Ron Prusia
-- Oct 2011
-- May 2019, simplified
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
entity UART_rx is
port (
  clk	   : in bit;   --Clock
  rx       : in std_logic; --Input serial data, must be same type as sync_ff
--
  data_out : out unsigned(7 downto 0);
  byte_rdy : out bit   --Flag to indicate the byte is ready to read
  );
end UART_rx;
--
architecture rtl of UART_rx is
--
-- Constants
--
  constant baud_divide : unsigned(15 downto 0) := x"01B1"; --50Mhz, 00D8/230,400, 01B1/115,200
--
-- Variables
--
  signal sync_ff      : unsigned(2 downto 0) := "111"; --Three registers to sync input data
  signal shft_reg     : unsigned(7 downto 0); --Input shift register
  signal shft_reg_en  : bit; --Shift register enable
  signal byte_rdy_ff  : bit; --Register output data ready
  signal byte_rdy_ff_d: bit;
--
  signal bit_cnt      : unsigned(2 downto 0);	--Output from mux to output register
  signal bit_cnt_sclr : bit; --Sync clear
  signal bit_cnt_en   : bit; --Count enable
  signal baud_cnt     : unsigned(15 downto 0); --Baud rate counter
  signal baud_cnt_sclr: bit; --Sync clear
--
  type ss_type is (init,start,bits_in,stop);
  signal ss_ps,ss_ns : ss_type;
--
-- Start of code, define outputs
--
  begin
  data_out <= shft_reg;
  byte_rdy <= byte_rdy_ff;
--
-- Clock processes
--
  clk_proc : process(clk,rx,byte_rdy_ff_d,ss_ns,shft_reg)
  begin
    if (clk'event AND clk = '1') then 
      sync_ff <= sync_ff(1 downto 0) & rx; --sync registers input
      byte_rdy_ff <= byte_rdy_ff_d;
      ss_ps <= ss_ns;   
--      
      if (shft_reg_en = '1') then shft_reg <= sync_ff(2) & shft_reg(7 downto 1); --shift right
      else shft_reg <= shft_reg;
      end if;
-- 
      if ((baud_cnt_sclr = '1') OR (baud_cnt = baud_divide)) then baud_cnt <= x"0000";
      else baud_cnt <= baud_cnt + 1;
      end if;
--
      if (bit_cnt_sclr = '1') then bit_cnt <= "000";
      elsif (bit_cnt_en = '1') then bit_cnt <= bit_cnt + 1;
      else bit_cnt <= bit_cnt;
      end if;
    end if;
  end process clk_proc;      
--
--   The state machine waits for the data ready signal it then proceeds to output
--   the bits. Bit 0 is read first. The output is high, until 
--
  comb_proc : process(ss_ps,ss_ns,bit_cnt,baud_cnt,sync_ff)
  begin
--
    bit_cnt_sclr <= '0'; --default state for variables is not asserted
    bit_cnt_en <= '0';
    baud_cnt_sclr <= '0';
    byte_rdy_ff_d <= '0';
    shft_reg_en <= '0';
--
    case ss_ps is
      when init => --Wait for falling edge
        bit_cnt_sclr <= '1';
        baud_cnt_sclr <= '1';
        if  (sync_ff(2) = '0') then ss_ns <= start;
        else ss_ns <= init;
        end if;
      when start => --Wait half a bit time
        if (baud_cnt(14 downto 0) = baud_divide(15 downto 1)) then
	      ss_ns <= bits_in;
	      baud_cnt_sclr <= '1';
	    else ss_ns <= start;
	    end if;
      when bits_in =>
        if (baud_cnt = baud_divide) AND (bit_cnt /= "111") then 
          ss_ns <= bits_in;
          shft_reg_en <= '1';  --Load bit
          bit_cnt_en <= '1'; --Increment bit count, baud cleared at cout in counter
        elsif (baud_cnt = baud_divide) AND (bit_cnt = "111") then 
          shft_reg_en <= '1';--load bit
          ss_ns <= stop;
        else ss_ns <= bits_in;
        end if;
      when stop =>
        if (baud_cnt = baud_divide) then 
          ss_ns <= init;
          byte_rdy_ff_d <= '1';
	    else ss_ns <= stop;
	    end if;
	  when others => ss_ns <= init;
	end case;
  end process comb_proc;	
end;
