--
--   Copyright (C) 2020 Ron Prusia
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program.  If not, see <http://www.gnu.org/licenses.
--
-- FIFO to data
--
-- March, 2020 - initial
-- June, 2023 - updated to support 1023 input bytes
--         
-- Ron Prusia
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
entity serial_datain is
port (
  clk       : in bit;						--Clock input 
  din       : in unsigned(15 downto 0); --
  fifo_used : in unsigned(10 downto 0);  --Number of 16-bit words
--
  fifo_rd   : out bit;
  dout      : out unsigned(15 downto 0)
  );
end serial_datain;
--
architecture rtl of serial_datain is
--
-- Variables
--
  signal word_cnt      : unsigned(8 downto 0); --Count 
  signal word_cnt_en   : bit;
  signal word_cnt_sclr : bit;
  signal len_ff        : unsigned(8 downto 0);
  signal len_ff_en     : bit;
--
-- State machine
--
  type ss_type is (init,setup1,setup2,sync1,sync2,rd1,rd2,rd3,rd4,wr1,wr2,wr3,wr4);
  signal ss_ps,ss_ns : ss_type;
--
begin
--
-- Clock processes, state machine and counter
--
  clk_proc : process(clk,ss_ns,word_cnt,word_cnt_en,word_cnt_sclr)
  begin
--  
    if (clk'event AND clk = '1') then 
      ss_ps <= ss_ns;
--
      if (len_ff_en = '1') then len_ff <= din(8 downto 0);
      else len_ff <= len_ff;
      end if;
--
      if (word_cnt_sclr = '1') then word_cnt <= "000000000";
      elsif (word_cnt_en = '1') then word_cnt <= word_cnt + 1;
      else word_cnt <= word_cnt;
      end if;
    end if;
--
  end process clk_proc;
--
-- Combinational process for state machine and output
--
  comb_proc : process (ss_ps,fifo_used,din,word_cnt,len_ff)
  begin
--
  word_cnt_en <= '0';    --default values
  word_cnt_sclr <= '0';
  word_cnt_en <= '0';
  fifo_rd <= '0';
  len_ff_en <= '0';
  dout <= x"0000";
--
  case ss_ps is 
    when init => --add one clock delay before setup
      word_cnt_sclr <= '1';
      ss_ns <= setup1;
    when setup1 =>  --Write out initialization 
      dout <= x"EB90";
      ss_ns <= setup2;
    when setup2 =>
      dout <= x"FE00";
      ss_ns <= sync1;
--
-- Waiting for sync and reading out sync places at least two null/blanks
-- on the dout bus.
--    
    when sync1 => --wait for data and sync word in fifo
      word_cnt_sclr <= '1';
      if (din = x"EB90") AND (fifo_used > 2) then --wait for at least three words in fifo
        fifo_rd <= '1';
        ss_ns <= sync2;
      elsif (din /= x"EB90") AND (fifo_used /=0) then --parse thru data looking for sync
        fifo_rd <= '1';
        ss_ns <= sync1;
      else ss_ns <= sync1;
      end if;
    when sync2 => --length/type available
      len_ff_en <= '1';
      if    (din(15 downto 9) = "0000001") then ss_ns <= wr1; -- force result for write
      elsif (din(15 downto 9) = "0000000") then ss_ns <= rd1; -- force result for read
      else ss_ns <= sync1;
      end if;
--
    when rd1 => --sync char
      dout <= x"EB90";
      ss_ns <= rd2;
    when rd2 => --type and length
      dout <= din;
      fifo_rd <= '1';
      ss_ns <= rd3;
    when rd3 => --address
      dout <= din;
      fifo_rd <= '1';
      ss_ns <= rd4;
    when rd4 => --wait length number of words
      if (len_ff = word_cnt) then
        ss_ns <= sync1;
      else
        word_cnt_en <= '1';
        ss_ns <= rd4;
      end if;
--
    when wr1 => --wait for enough write data in fifo
      if  (fifo_used >= (("00" & len_ff)+2)) then
        dout <= x"EB90";
        ss_ns <= wr2;
      else ss_ns <= wr1;
      end if;
    when wr2 => --type/length from fifo
      dout <= din;
      fifo_rd <= '1';
      ss_ns <= wr3;
    when wr3 => --address  
      dout <= din;
      fifo_rd <= '1';
      ss_ns <= wr4;
    when wr4 =>
      dout <= din;
      fifo_rd <= '1';
      word_cnt_en <= '1';
      if (word_cnt = len_ff) then ss_ns <= sync1;
      else ss_ns <= wr4;
      end if;
    when others => ss_ns <= init;
  end case;   
--    	    
  end process comb_proc;
end;
