--
--   Copyright (C) 2020 Ron Prusia
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program.  If not, see <http://www.gnu.org/licenses.
--
-- FIFO to data
--
-- April, 2020
--         
-- Ron Prusia
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
entity fifo_load is
port (
  clk       : in bit;						--Clock input 
  din       : in unsigned(15 downto 0); --
--
  fifo_wr   : out bit;
  dout      : out unsigned(15 downto 0)
  );
end fifo_load;
--
architecture rtl of fifo_load is
--
-- Variables
--
  signal data_ff       : unsigned(15 downto 0);--holds input data for one clock
  signal word_cnt      : unsigned(8 downto 0); --Count 
  signal word_cnt_en   : bit;
  signal word_cnt_sclr : bit;
  signal len_ff        : unsigned(8 downto 0);
  signal len_ff_en     : bit;
--
-- State machine
--
  type ss_type is (init,mes_len,wr,flag);
  signal ss_ps,ss_ns : ss_type;
--
begin
--
  dout <= data_ff;
--
-- Clock processes, state machine and counter
--
  clk_proc : process(clk,ss_ns,word_cnt,word_cnt_en,word_cnt_sclr)
  begin
--  
    if (clk'event AND clk = '1') then 
      ss_ps <= ss_ns;
      data_ff <= din;
--
      if (len_ff_en = '1') then len_ff <= '0' & data_ff(7 downto 0);
      else len_ff <= len_ff;
      end if;
--
      if (word_cnt_sclr = '1') then word_cnt <= "000000000";
      elsif (word_cnt_en = '1') then word_cnt <= word_cnt + 1;
      else word_cnt <= word_cnt;
      end if;
    end if;
--
  end process clk_proc;
--
-- Combinational process for state machine and output
--
  comb_proc : process (ss_ps,din,word_cnt,len_ff,data_ff)
  begin
--
  word_cnt_en <= '0';    --default values
  word_cnt_sclr <= '0';
  fifo_wr <= '0';
  len_ff_en <= '0';
--
  case ss_ps is 
    when init => --add one clock delay before setup
      word_cnt_sclr <= '1';
      if (data_ff = x"EB90") AND (din(15 downto 9)="1111110") then --flag
        fifo_wr <= '1';
        ss_ns <= flag;
      elsif (data_ff = x"EB90") AND (din(15 downto 9)="0000000") then --read
        fifo_wr <= '1';
        ss_ns <= mes_len;
      else ss_ns <= init;
      end if;
    when mes_len =>  --Write out initialization 
      fifo_wr <= '1';
      len_ff_en <= '1';
      ss_ns <= wr;
--
    when wr => --wait for enough write data in fifo
      fifo_wr <= '1';
      word_cnt_en <= '1';
      if (len_ff = word_cnt) then ss_ns <= init;
      else ss_ns <= wr;
      end if;
    when flag => --flag is two word command
      fifo_wr <= '1';
      ss_ns <= init;
    when others => ss_ns <= init;
  end case;   
--    	    
  end process comb_proc;
end;
