-- 
--   Copyright (C) 2023 Ron Prusia
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program.  If not, see <http://www.gnu.org/licenses.
--
-- Ethernet UDP Receiver
--
-- Read UDP data and compare.
--
-- The external PROM file indicates the expect value. If the upper bit is set,
-- then the value is compared. Otherwise the value is ignored.
--
-- The UDP data format is shown below, addresses are in hex:
--  0 to 6 		: 0x55 - Preamble
--	7			: 0xD5 - Preamble
--  8 to D  	: Don't care - My MAC
--  E to 13 	: Don't care - Computer mac
--  14      	: 0x08 - Ethernet type
--  15      	: 0x00 - Ethernet type
--  16 to 17	: 0x4500 - IP header
--  18 to 19    : Don't care - type of service, length
--  1A to 1D	: Don't care - Ident 16-bits, flash/fragment offset
--  1E 			: Don't care - Time to live
--  1F          : 0x11 - UDP protocol
--  20 to 21	: Don't care - IP checksum
--  22 to 25    : Don't care - Computer IP address
--  26 to 29    : 239.255.x.x - My IP address (Only upper two bytes checked)
--  2A to 2B    : Don't care - Source port
--  2C to 2D    : 0x15C0 - Destination port
--  2E to 2F    : Don't care - UDP length
--  30 to 31    : Don't care - UDP checksum
--  32 to 33    : 0x0010 - Preamble size (Start of UDP packet 0/1)
--  34 to 35    : 0x0000 - Post-amble size
--  36 to 41    : 0x41 0x53 0x43 0x2d 0x45 0x31 0x2e 0x31 0x37 0x00 0x00 0x00 - ACN ID
--  42 to 43    : Don't care - Flags and length of data
--  44 to 47    : 0x0000 0004 - Vector
--  48 to 57    : Don't care - Customer ID
--  58 to 59    : Don't care - Flags and length of data
--  5A to 5D    : 0x0000 0002 - Vector
--  5E to 9E    : Don't care - Source name 64 bytes
--  9F          : Don't care - Priority
--  A0 to A1    : Don't care - Reserved
--  A2          : Don't care - Sequence Number
--  A3          : Store - Option flags (Bits 7,6 7-preview (ignore packet), 6-turn off universe)
--  A3 to A4    : Store - Universe number
--  A5 to A6    : Don't care - Flags and length
--  A7          : 0x02 - Vector
--  A8          : 0xA1  - Address type and data type
--  A9 to AA    : 0x0000 - First property address
--  AB to AC    : 0x0001 - Address increment
--  AD to AE    : Store - Property value count
--  AF to end   : Store - Packet data, plus start character
--
-- The universe data is written to memory for output. If the output is valid, the upper bit, eighth
-- bit is set in the memory.
--
-- Oct 2023  - Initial
--         
-- Ron Prusia
--
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
--
entity rx_ctrl is
port (
	rx_clk     	: in std_logic;						--Clock input 25 Mhz
	rx_data		: in std_logic_vector(7 downto 0);	--Ethernet data
	rx_dv       : in std_logic;						--Data valid, asserted every four clocks
--
	fifo_dout	: out std_logic_vector (7 downto 0);--fifo data
	fifo_wr     : out std_logic;                    --fifo write
	bus_addr_out    : out std_logic_vector(15 downto 0);
	bus_byte_cnt_out: out std_logic_vector(9 downto 0);
	rx_ok_flag  : out std_logic
	);
end rx_ctrl;
--
architecture rtl of rx_ctrl is
--
-- Variables
--
	signal rx_valid_flag    : std_logic;  --Flag that indicates input UDP stream good
	signal prom_valid_flag  : std_logic;  --Flag that indicates valid data in prom
--
	signal univ_ff	      	: std_logic_vector (15 downto 0); --Latch universe number
	signal bus_addr         : std_logic_vector (15 downto 0);
	signal prop_cnt_ff      : std_logic_vector (9 downto 0);          --Latch channel count, from UDP length
	signal bus_byte_cnt     : std_logic_vector (9 downto 0);
--
	signal ss_cnt           : std_logic_vector (9 downto 0);--State machine counter
	signal ss_cnt_sclr      : std_logic; --Synchronous clear
	signal ss_cnt_en        : std_logic; --Count enable
--
-- State machine
--
	type ss_type is (init,check,store,wait_end);
    signal ss_ps,ss_ns : ss_type;
--
begin
--
-- Output values
--
	bus_addr_out <= bus_addr;
	bus_byte_cnt_out <= bus_byte_cnt;
	rx_ok_flag <= rx_valid_flag;
--
-- Asserted rx_valid_flag if the input Ethernet stream matches the expected data.
-- Didn't check all of the bytes.
--
	valid_data_proc : process(rx_data,ss_cnt)
    begin
      rx_valid_flag <= '0';
      case ss_cnt is
        when "0000000000" => if rx_data=x"55" then rx_valid_flag <= '1'; end if;--preamble
        when "0000000001" => if rx_data=x"55" then rx_valid_flag <= '1'; end if;
        when "0000000010" => if rx_data=x"55" then rx_valid_flag <= '1'; end if;
        when "0000000011" => if rx_data=x"55" then rx_valid_flag <= '1'; end if;
        when "0000000100" => if rx_data=x"55" then rx_valid_flag <= '1'; end if;
        when "0000000101" => if rx_data=x"55" then rx_valid_flag <= '1'; end if;
        when "0000000110" => if rx_data=x"55" then rx_valid_flag <= '1'; end if;
        when "0000000111" => if rx_data=x"D5" then rx_valid_flag <= '1'; end if;--preamble end
        when "0000010100" => if rx_data=x"08" then rx_valid_flag <= '1'; end if;--mac type
        when "0000010101" => if rx_data=x"00" then rx_valid_flag <= '1'; end if;
        when "0000010110" => if rx_data=x"45" then rx_valid_flag <= '1'; end if;--ip
        when "0000010111" => if rx_data=x"00" then rx_valid_flag <= '1'; end if;
        when "0000011111" => if rx_data=x"11" then rx_valid_flag <= '1'; end if;--udp packet
        when "0000100110" => if rx_data=x"EF" then rx_valid_flag <= '1'; end if;--ip address, multicast
        when "0000100111" => if rx_data=x"FF" then rx_valid_flag <= '1'; end if;
        when "0000101100" => if rx_data=x"15" then rx_valid_flag <= '1'; end if;--dest port
        when "0000101101" => if rx_data=x"C0" then rx_valid_flag <= '1'; end if;
        when "0000110010" => if rx_data=x"00" then rx_valid_flag <= '1'; end if;--start of ACN message
        when "0000110011" => if rx_data=x"10" then rx_valid_flag <= '1'; end if;--just check first 4 bytes of ACN message
        when "0000110100" => if rx_data=x"00" then rx_valid_flag <= '1'; end if;
        when "0000110101" => if rx_data=x"00" then rx_valid_flag <= '1'; end if; 
        when "0010100111" => if rx_data=x"02" then rx_valid_flag <= '1'; end if;--vector
        when "0010101000" => if rx_data=x"A1" then rx_valid_flag <= '1'; end if;--address type
        when others => rx_valid_flag <= '1';--nominally data is OK
      end case;
    end process valid_data_proc;
--
-- Registers
--
	reg_ff_proc : process(rx_clk,rx_data,ss_cnt,rx_dv,ss_ps,ss_ns)
	begin
		if (rx_clk'event and rx_clk = '1') then
--
          ss_ps <= ss_ns;        
--
	      if (ss_cnt_sclr = '1') then ss_cnt <= "0000000000";	--sclr
		  elsif (ss_cnt_en = '1') then ss_cnt <= ss_cnt + 1;--count enable
	      else ss_cnt <= ss_cnt; --else hold count value
	      end if;
--	      
	      if (ss_cnt = "0010100100") AND (ss_ps = check) then --latch universe lsb
            univ_ff(7 downto 0) <= rx_data; --count = 0xA4
	      else univ_ff(7 downto 0) <= univ_ff(7 downto 0);
	      end if;
--	      
	      if (ss_cnt = "0010100011") AND (ss_ps = check) then --latch universe msb
            univ_ff(15 downto 8) <= rx_data;--count = 0xA3
	      else univ_ff(15 downto 8) <= univ_ff(15 downto 8);
	      end if;
--
   	      if (ss_cnt = "0000101111") AND (ss_ps = check) then --latch byte count
            prop_cnt_ff(7 downto 0) <= rx_data;--count = 0x2F
	      else prop_cnt_ff(7 downto 0) <= prop_cnt_ff(7 downto 0);
	      end if;
--	      
	      if (ss_cnt = "0000101110") AND (ss_ps = check) then --latch byte count
            prop_cnt_ff(9 downto 8) <= rx_data(1 downto 0);--count = ox2E
	      else prop_cnt_ff(9 downto 8) <= prop_cnt_ff(9 downto 8);
	      end if;	      
        end if;
	end process reg_ff_proc;
--
-- State machine process to compare input to PROM.
-- Also outputs test sequence or clears output if selected from flags.
--
	ss_comb_proc : process(ss_ps,ss_ns,ss_cnt,rx_dv,rx_data,rx_valid_flag,prop_cnt_ff,bus_byte_cnt,univ_ff,bus_addr)
	begin
		ss_cnt_en <= '0';		--Set default values for all variables to deasserted or zero
		ss_cnt_sclr <= '0';		--Variables are only in case statements where asserted.
		fifo_wr <= '0';			--Write to universe memory
--
-- fifo data out based on state machine count
-- output header before input data, pulling it out of state machine allows it to be asserted longer for simulation
--
        if ss_ps = check then
          case ss_cnt is
	        when "0010101010" => fifo_dout <= x"90";
	        when "0010101011" => fifo_dout <= x"EB";
	        when "0010101100" => fifo_dout <= bus_byte_cnt(8 downto 1);
	        when "0010101101" => fifo_dout <= "0000001" & bus_byte_cnt(9);
	        when "0010101110" => fifo_dout <= x"00";
   	        when "0010101111" => fifo_dout <= bus_addr(7 downto 0);
   	        when others => fifo_dout <= rx_data; -- input data
          end case;
        else fifo_dout <= rx_data;
	    end if;
--
-- Calculate byte count from UDP length, add one if odd.
-- 
		if (prop_cnt_ff(0) = '0') then bus_byte_cnt <= prop_cnt_ff - "0010000110"; --odd count, even data (86hex overhead, dam s/w engineers)
	    else bus_byte_cnt <= prop_cnt_ff - "0010000101"; --even count - odd, extra byte added for word output, last byte CRC
	    end if;
--
--  Calculate bus address from universe, shift right in output   
--
		bus_addr <= (univ_ff - 1);
--
-- State machine next state and outputs 
-- Rx_dv is only asserted every four clocks, make decisons only on this signal asserted.
--
		case ss_ps is
		when init =>            --Wait for start of message
		  if (rx_dv = '1') then
            ss_cnt_en <= '1';
            ss_ns <= check;
          else
   		    ss_cnt_sclr <= '1';   --Clear counter		  
		    ss_ns <= init;
		  end if;
--
-- The bus header of six bytes is prewritten while counting through the last six bytes of the 
-- ACN 131 header.
--		  
		when check =>           --Count through message verifing data
		  if (rx_dv = '1') and (rx_valid_flag = '1') AND (ss_cnt="0010101111") then --end of testing input packet, and byte#6 (address upper)
            fifo_wr <= '1';
            ss_cnt_sclr <= '1';
            ss_ns <= store;
          elsif (rx_dv = '1') and (rx_valid_flag = '1') and (ss_cnt="0010101110") then --write out bus header byte#5 (address lower)
            fifo_wr <= '1';            
            ss_cnt_en <= '1';
            ss_ns <= check;    
          elsif (rx_dv = '1') and (rx_valid_flag = '1') and (ss_cnt="0010101101") then --write out bus header byte#4 (length upper), write bit
            fifo_wr <= '1';            
            ss_cnt_en <= '1';
            ss_ns <= check;
          elsif (rx_dv = '1') and (rx_valid_flag = '1') and (ss_cnt="0010101100") then --write out bus header byte#3 (length lower, in words half value)
            fifo_wr <= '1';            
            ss_cnt_en <= '1';
            ss_ns <= check;
          elsif (rx_dv = '1') and (rx_valid_flag = '1') and (ss_cnt="0010101011") then --write out bus header byte#2 (sync upper)
            fifo_wr <= '1';            
            ss_cnt_en <= '1';
            ss_ns <= check;
          elsif (rx_dv = '1') and (rx_valid_flag = '1') and (ss_cnt="0010101010") then --write out bus header byte#1 (sync lower)
            if univ_ff > 512 then 
              ss_cnt_sclr <= '1';
              ss_ns <= wait_end; --if address greater than 512 ignore packet
            else 
              fifo_wr <= '1';            
              ss_cnt_en <= '1';            
              ss_ns <= check;
            end if;            
          elsif (rx_dv = '1') and (rx_valid_flag = '1') then
            ss_cnt_en <= '1';
            ss_ns <= check;
          elsif (rx_dv = '1') and (rx_valid_flag = '0') then
   		    ss_cnt_sclr <= '1';   --Clear counter		  
		    ss_ns <= wait_end;
		  else ss_ns <= check;
		  end if;
		when store =>			  --Store input data into memory, ss_cnt is addr
		  if (rx_dv = '1') then
            ss_cnt_en <= '1';
            fifo_wr <= '1';
          end if;
          if (ss_cnt = "1000000000") or (ss_cnt = bus_byte_cnt) then
            ss_cnt_sclr <= '1';
            ss_ns <= wait_end;--ends at 512 or end of byte count (minus 1/2 for count at zero)
	      else ss_ns <= store;
	      end if;
		when wait_end =>		  --Wait four clocks to rx_dv not to be asserted
          if (rx_dv = '1') then 
            ss_cnt_sclr <= '1';
            ss_ns <= wait_end;
	      elsif (ss_cnt > 4) then
            ss_ns <= init;
	      else 
	        ss_cnt_en <= '1';
	        ss_ns <= wait_end;
	      end if;
	    when others => ss_ns <= init;
	    end case;      	    
	end process ss_comb_proc;
end;
