-- 
--   Copyright (C) 2015 Ron Prusia
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program.  If not, see <http://www.gnu.org/licenses.
-- 
-- Converts the 2 bits to bytes
--
-- October 2023
--

-- Ron Prusia
--
--Library
library ieee;
--Package Statement
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
--
--
entity rx_byte is
--
  port (
    clk       : in std_logic;
    rx        : in std_logic_vector(1 downto 0);
    rx_dv     : in std_logic;
--
    rx_out    : out std_logic_vector(7 downto 0);
    rx_dv_out : out std_logic
    );
  end rx_byte;
--
  architecture rtl of rx_byte is
--
-- Signal declarations
--
  signal sync1_ff     : std_logic_vector (2 downto 0);
  signal sync2_ff     : std_logic_vector (2 downto 0);
  signal sync3_ff     : std_logic_vector (2 downto 0);
--
  signal data_ff_en1 : std_logic;
  signal data_ff_en2 : std_logic;
  signal data_ff_en3 : std_logic;
  signal data_ff_en4 : std_logic;
  signal data_ff_sclr: std_logic;
  signal data_ff     : std_logic_vector (7 downto 0);
--
  signal rx_dv_ff    : std_logic;
  signal rx_dv_ff_d  : std_logic;
--
-- State machine
--
  type ss_type is (init,bits1,bits2,bits3,bits4);
  signal ss_ps,ss_ns : ss_type;
--
begin
--
  rx_out <= data_ff;
  rx_dv_out <= rx_dv_ff;
--
-- Clock processes
--
  clock_proc : process(clk,sync1_ff,sync2_ff,sync3_ff,rx,rx_dv)
  begin 
    if (clk'event and clk = '1') then
      ss_ps <= ss_ns;  --state machine
--
      sync1_ff(0) <= rx(0); --sync inputs
      sync1_ff(1) <= rx(1);
      sync1_ff(2) <= rx_dv;			
      sync2_ff <= sync1_ff;
      sync3_ff <= sync2_ff;
--      
      rx_dv_ff <= rx_dv_ff_d; --register output, delay one clock so data is valid
--
-- Register input data
--
      if (data_ff_sclr = '1') then data_ff(1 downto 0) <= "00";
      elsif (data_ff_en1 = '1') then data_ff(1 downto 0) <= sync3_ff(1 downto 0); --load two bits
      else data_ff(1 downto 0) <= data_ff(1 downto 0);
      end if;
      if (data_ff_sclr = '1') then data_ff(3 downto 2) <= "00";       
      elsif (data_ff_en2 = '1') then data_ff(3 downto 2) <= sync3_ff(1 downto 0);
      else data_ff(3 downto 2) <= data_ff(3 downto 2);
      end if;
      if (data_ff_sclr = '1') then data_ff(5 downto 4) <= "00"; 
      elsif (data_ff_en3 = '1') then data_ff(5 downto 4) <= sync3_ff(1 downto 0);
      else data_ff(5 downto 4) <= data_ff(5 downto 4);
      end if; 
      if (data_ff_sclr = '1') then data_ff(7 downto 6) <= "00";
      elsif (data_ff_en4 = '1') then data_ff(7 downto 6) <= sync3_ff(1 downto 0);
      else data_ff(7 downto 6) <= data_ff(7 downto 6);
      end if;  
    end if;
  end process clock_proc;
--
-- Next state and output variables combinational logic. This is state machine as described 
-- in the header.
--
  ss_comb_proc : process(ss_ps,ss_ns,sync3_ff)
  begin
    data_ff_en1 <= '0';   --Set default values for all variables to deasserted or zero
    data_ff_en2 <= '0';   --Variables are only in case statements where asserted.
    data_ff_en3 <= '0';
    data_ff_en4 <= '0';
    data_ff_sclr <= '0';
    rx_dv_ff_d <= '0';
--
    case ss_ps is
      when init => --Power up
        if (sync3_ff = "101") then 
          data_ff_en1 <= '1';
          ss_ns <= bits2;
        else 
          data_ff_sclr <= '1';
          ss_ns <= init;
        end if;
      when bits1 =>           --Check for data valid
        if sync3_ff(2) = '1' then 
          data_ff_en1 <= '1';	--Load bits 0/1
          ss_ns <= bits2;
	     else ss_ns <= init;
        end if;
      when bits2 =>           --Check for data valid
        if sync3_ff(2) = '1' then 
          data_ff_en2 <= '1';	--Load bits 3/2
          ss_ns <= bits3;
        else ss_ns <= init;
        end if;
      when bits3 =>           --Check for data valid
        if sync3_ff(2) = '1' then 
          data_ff_en3 <= '1';	--Load bits 5/4
          ss_ns <= bits4;
        else ss_ns <= init;
        end if;
      when bits4 =>           --Check for data valid
        if sync3_ff(2) = '1' then
          rx_dv_ff_d <= '1';	--assert data valid
          data_ff_en4 <= '1';	--Load bits 7/6
          ss_ns <= bits1;
        else ss_ns <= init;
        end if;
    end case;
  end process ss_comb_proc;
end;
