--
--   Copyright (C) 2020 Ron Prusia
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program.  If not, see <http://www.gnu.org/licenses.
--
-- Identification, must be first module in stream
--
-- March, 2020
--         
-- Ron Prusia
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
entity eth_acn_id_min is
port (
  clk       : in bit;						--Clock input 
  din       : in unsigned(15 downto 0); --
--
  dout      : out unsigned(15 downto 0)
  );
end eth_acn_id_min;
--
architecture rtl of eth_acn_id_min is
--
-- Constants
--
  constant mod_id   : unsigned(15 downto 0) := x"FFF1"; --module identification
  constant mod_len  : unsigned(15 downto 0) := x"0004"; --length in words
  constant mod_ver1 : unsigned(15 downto 0) := x"0629"; --Version Month/Day
  constant mod_ver2 : unsigned(15 downto 0) := x"2023"; --Version Year

--
-- Variables
--
  signal mod_ff        : unsigned(15 downto 0);
  signal cmd_len_ff    : unsigned(8 downto 0);
  signal cmd_len_ff_en : bit;
  signal cmd_len_ff_dec: bit;
--
-- State machine
--
  type ss_type is (init,decode1,decode2,addr,read0,read1,read2,read3,read4);
  signal ss_ps,ss_ns : ss_type;
--
begin
--
-- Clock processes, state machine and counter
--
  clk_proc : process(clk,ss_ns,mod_ff)
  begin
--  
    if (clk'event AND clk = '1') then 
      ss_ps <= ss_ns;
      mod_ff <= din; --hold input data one clock
--
      if (cmd_len_ff_en = '1') then cmd_len_ff <= mod_ff(8 downto 0);
      elsif (cmd_len_ff_dec = '1') then cmd_len_ff <= cmd_len_ff - 1;
      else cmd_len_ff <= cmd_len_ff;
      end if;
    end if;
--
  end process clk_proc;
--
-- Combinational process for state machine and output
--
  comb_proc : process (ss_ps,mod_ff,din,cmd_len_ff)
  begin
--
  dout <= mod_ff;        --output defaults to din, delayed one clock
  cmd_len_ff_en <= '0';
  cmd_len_ff_dec <= '0';
--
-- Initial sync word is defined as 0x0000 followed by 0xEB90
-- All logic is defined off the mod_ff input
--
  case ss_ps is 
    when init => --add one clock delay before setup
      if (mod_ff = x"0000") and (din = x"EB90") then ss_ns <= decode1;
      else ss_ns <= init;
      end if; 
    when decode1 => --wait one state for sync to be in mod_ff
      ss_ns <= decode2;
    when decode2 => --mod_ff equals command
      cmd_len_ff_en <= '1';
      if (mod_ff(15 downto 9) = "1111111") then ss_ns <= addr; --addr command
      elsif (mod_ff(15 downto 9) = "0000000") AND (mod_ff(8 downto 0) /= "000000000") then ss_ns <= read0; --read command
      else ss_ns <= init; --all other commands are ignored.
      end if;
--
-- Address function
--    
    when addr => --mod_ff equals address of command
      dout <= x"0004"; --increment address
      ss_ns <= init;
--
-- Write function
--    
    when read0 => --mod_ff equals address of command
      if    mod_ff = x"0000" then ss_ns <= read1;
      elsif mod_ff = x"0001" then ss_ns <= read2;
      elsif mod_ff = x"0002" then ss_ns <= read3;
      elsif mod_ff = x"0003" then ss_ns <= read4;
      else ss_ns <= init;
      end if;
    when read1 =>
      dout <= mod_id;
      if (cmd_len_ff = "000000000") then ss_ns <= init;
      else 
        cmd_len_ff_dec <= '1';
        ss_ns <= read2;
      end if;
    when read2 =>
      dout <= mod_len;
      if (cmd_len_ff = "000000000") then ss_ns <= init;
      else 
        cmd_len_ff_dec <= '1';
        ss_ns <= read3;
      end if;      
    when read3 =>
      dout <= mod_ver1;
      if (cmd_len_ff = "000000000") then ss_ns <= init;
      else ss_ns <= read4;
      end if;
    when read4 =>
      dout <= mod_ver2;
      ss_ns <= init;
    when others => ss_ns <= init;
  end case;   
--    	    
  end process comb_proc;
end;
