--
--   Copyright (C) 2020 Ron Prusia
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program.  If not, see <http://www.gnu.org/licenses.
--
-- Pixel Controller
--
-- March, 2020
-- June, 2023 updated for fifo ack
--         
-- Ron Prusia
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
entity ws2811_ctrl is
generic (
  period_count  : integer := 62;
  high_cnt      : integer := 35;
  low_cnt       : integer := 18
  );
port (
  clk           : in bit;					--Clock input
  fifo_empty    : in bit;
  mem_in        : in unsigned(7 downto 0);  --Input from memory
--
  fifo_ack      : out bit;
  pixel_out     : out bit
  );
end ws2811_ctrl;
--
architecture rtl of ws2811_ctrl is
--
-- Variables
--
  signal per_cnt        : unsigned(7 downto 0);
  signal per_cnt_sclr   : bit;
  signal bit_cnt        : unsigned(2 downto 0);
  signal bit_cnt_en     : bit;
  signal bit_cnt_sclr   : bit;
  signal shift_reg      : unsigned(7 downto 0);
  signal shift_reg_ld   : bit;
  signal shift_reg_shft : bit;
  signal pixel_ff,pixel_ff_d : bit;
--
-- State machine
--
  type ss_type is (init,delay,bit_low,bit_high);
  signal ss_ps,ss_ns : ss_type;
--  
begin
  pixel_out <= pixel_ff;
--
-- Clock processes, state machine and counter
--
  clk_proc : process(clk)
  begin
--  
    if (clk'event) AND (clk = '1') then 
      ss_ps <= ss_ns;
      pixel_ff <= pixel_ff_d;
--    
      if (per_cnt_sclr = '1') then per_cnt <= x"00";
      else per_cnt <=  per_cnt + 1;
      end if;
--
      if (bit_cnt_sclr = '1') then bit_cnt <= "000";
      elsif (bit_cnt_en = '1') then bit_cnt <= bit_cnt  + 1;
      else bit_cnt <= bit_cnt;
      end if;
--
      if (shift_reg_ld = '1') then shift_reg <= mem_in;
      elsif (shift_reg_shft = '1') then shift_reg <= shift_reg(6 downto 0) & '0';
      else shift_reg <= shift_reg;
      end if;      
    end if;
    
--
  end process clk_proc;
--
-- Combinational process
--
  comb_proc : process(clk,ss_ps,fifo_empty,shift_reg,bit_cnt,per_cnt)
  begin
--
  per_cnt_sclr <= '0';    --default values
  bit_cnt_sclr <= '0';
  bit_cnt_en <= '0';
  fifo_ack <= '0';
  shift_reg_ld <= '0';
  shift_reg_shft <= '0';
  pixel_ff_d <= '0';
--
-- State machine; wait for write to memory
-- Load, shift thru bits, increment
--
  case ss_ps is   
    when init => --wait for write to module
      per_cnt_sclr <= '1';
      bit_cnt_sclr <= '1';
      if (fifo_empty = '0') then ss_ns <= delay;
      else ss_ns <= init;
      end if;
    when delay =>
      pixel_ff_d <= '1';
      shift_reg_ld <= '1';
      fifo_ack <= '1';
      ss_ns <= bit_high;
    when bit_high => 
      pixel_ff_d <= '1';
      if ((shift_reg(7) = '1') AND (per_cnt = high_cnt - 1)) OR
         ((shift_reg(7) = '0') AND (per_cnt = low_cnt - 1)) then ss_ns <= bit_low;
      else 
        ss_ns <= bit_high;
      end if;
    when bit_low =>
      if (per_cnt = period_count - 1) AND (bit_cnt = "111") AND (fifo_empty = '1') then
        ss_ns <= init;
      elsif (per_cnt = period_count - 1) AND (bit_cnt = "111") then
        per_cnt_sclr <= '1';
        shift_reg_ld <= '1';
        fifo_ack <= '1';
        bit_cnt_sclr <= '1';
        ss_ns <= bit_high;
      elsif (per_cnt = period_count - 1) then
        per_cnt_sclr <= '1';
        shift_reg_shft <= '1';
        bit_cnt_en <= '1';
        ss_ns <= bit_high;
      else ss_ns <= bit_low;
      end if;
    when others => ss_ns <= init;
  end case;       
  end process comb_proc;  
end;
