--
--   Copyright (C) 2020 Ron Prusia
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program.  If not, see <http://www.gnu.org/licenses.
--
--  Output controller for DMX, wait for data in fifo
--
-- June, 2023 baseline
--         
-- Ron Prusia
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
entity dmx_ctrl is
port (
  clk           : in bit;					--Clock input
  fifo_empty    : in bit;
  mem_in        : in unsigned(7 downto 0);  --Input from memory
--
  fifo_ack      : out bit;
  dmx_out     : out bit
  );
end dmx_ctrl;
--
architecture rtl of dmx_ctrl is
--
-- Variables
--
  signal clk_cnt        : unsigned(7 downto 0); --divide 50mhz by 200, for 250Kbit data rate
  signal clk_cnt_cout   : bit;
  signal bit_cnt        : unsigned(4 downto 0);
  signal bit_cnt_en     : bit;
  signal bit_cnt_sclr   : bit;
  signal shift_reg      : unsigned(10 downto 0); --one start, two stop bits
  signal shift_reg_ld   : bit;
  signal shift_reg_shft : bit;
  signal shift_reg_start: bit;
  signal pixel_ff,pixel_ff_d : bit; -- output is registered
--
-- State machine
--
  type ss_type is (init,break,mab,dataout);
  signal ss_ps,ss_ns : ss_type;
--  
begin
  dmx_out <= pixel_ff;
--
-- Clock processes, state machine and counter
--
  clk_proc : process(clk)
  begin
--  
    if (clk'event) AND (clk = '1') then 
      ss_ps <= ss_ns;
      pixel_ff <= pixel_ff_d;
--    
      if (clk_cnt = 199) then --generate 250Kbit enable
         clk_cnt <= x"00";
         clk_cnt_cout <= '1';
      else
        clk_cnt_cout <= '0'; 
        clk_cnt <=  clk_cnt + 1;
      end if;
--
      if (bit_cnt_sclr = '1') then bit_cnt <= "00000";
      elsif (bit_cnt_en = '1') then bit_cnt <= bit_cnt  + 1;
      else bit_cnt <= bit_cnt;
      end if;
--
      if (shift_reg_ld = '1') then shift_reg <= "11" & mem_in & '0';
      elsif (shift_reg_start = '1') then shift_reg <= "11000000000"; --start word
      elsif (shift_reg_shft = '1') then shift_reg <= '0' & shift_reg(10 downto 1); --lsb is output
      else shift_reg <= shift_reg;
      end if;      
    end if;
    
--
  end process clk_proc;
--
-- Combinational process
--
  comb_proc : process(clk,ss_ps,fifo_empty,shift_reg,bit_cnt,clk_cnt_cout)
  begin
--
  bit_cnt_sclr <= '0';    --default values
  bit_cnt_en <= '0';
  fifo_ack <= '0';
  shift_reg_ld <= '0';
  shift_reg_start <= '0';
  shift_reg_shft <= '0';
  pixel_ff_d <= '0';
--
-- State machine; wait for write to memory
-- Load, shift thru bits, increment
--
  case ss_ps is   
    when init => --wait for write to module
      pixel_ff_d <= '1';
      bit_cnt_sclr <= '1';
      if (fifo_empty = '0') AND (clk_cnt_cout = '1') then ss_ns <= break;
      else ss_ns <= init;
      end if;
    when break => --output low default, break 92-96us
       if (clk_cnt_cout = '1') AND (bit_cnt = "10111") then
         bit_cnt_sclr <= '1';
         ss_ns <= mab;
       elsif (clk_cnt_cout = '1') then
         bit_cnt_en <= '1';
         ss_ns <= break;
       else ss_ns <= break;
       end if;
    when mab => --output high 12uS
        pixel_ff_d <= '1';
      if (clk_cnt_cout = '1') AND (bit_cnt = "00010") then
         bit_cnt_sclr <= '1';
         shift_reg_start <= '1';
         ss_ns <= dataout;
       elsif (clk_cnt_cout = '1') then
         bit_cnt_en <= '1';
         ss_ns <= mab;
       else ss_ns <= mab;
       end if;   
    when dataout => --output data till fifo empty
      if shift_reg(0) = '1' then pixel_ff_d <= '1';
      end if;
      if (fifo_empty = '1') AND (bit_cnt = "01010") AND (clk_cnt_cout = '1') then
        ss_ns <= init;
      elsif (bit_cnt = "01010") AND (clk_cnt_cout = '1') then
        ss_ns <= dataout;
        bit_cnt_sclr <= '1';
        shift_reg_ld <= '1';
        fifo_ack <= '1';
      elsif (clk_cnt_cout = '1') then
        ss_ns <= dataout;
        bit_cnt_en <= '1';
        shift_reg_shft <= '1';
      else ss_ns <= dataout;
      end if;
    when others => ss_ns <= init;
  end case;       
  end process comb_proc;  
end;
