--
--   Copyright (C) 2024 Ron Prusia
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program.  If not, see <http://www.gnu.org/licenses.
--
-- I2C master bus controller.
-- Single controller on bus, no 10_bit address support or clock stretching
-- Memory Map
-- 0x00 - ID
-- 0x01 - Length
-- 0x02 - reg_addr/dev_addr
-- 0x03 - control register
--        bit 0 - asserted for write (cleared at end of write)
--        bit 1 - asserted for read (cleared at end of read)
--        bit 2 to 4 - number of bytes to read or write (0 is 1, and 7 is eight)
--        bit 6,7 - spare
--        bit 8,9 - clock speed (00-100khz, 01-400khz, 10 and 11-1MHz)
--        bit 10 to 14 - spare
--        bit 15 - nack error
-- 0x04 - byte2/byte1 read/write
-- 0x05 - byte5/byte3 read/write
-- 0x06 - byte6/byte5 read/write
-- 0x07 - byte8/byte7 read/write
--
-- March, 2024
--         
-- Ron Prusia
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
entity mas_i2c is
port (
  clk        : in bit;						--Clock input 
  sda_in     : in std_logic;
  din1       : in unsigned(15 downto 0); --address 0x03
  din2       : in unsigned(15 downto 0); --address 0x04
  din3       : in unsigned(15 downto 0); --address 0x05
  din4       : in unsigned(15 downto 0); --address 0x06
  din5       : in unsigned(15 downto 0); --address 0x07
  din6       : in unsigned(15 downto 0); --address 0x08
--
  scl_out    : out std_logic;
  sda_out    : out std_logic;
  dout       : out unsigned(7 downto 0);
  reg_wr     : out bit;
  reg_sel    : out unsigned(2 downto 0);
  clr_read   : out bit;
  clr_write  : out bit;
  set_nack_err : out bit
  );
end mas_i2c;
--
architecture rtl of mas_i2c is
--
-- Variables
--
  signal clk_cnt            : unsigned(9 downto 0);
  signal clk_cnt_sclr       : bit;
  signal clk_cnt_cout       : bit;
  signal bit_cnt            : unsigned(3 downto 0); --Count 
  signal bit_cnt_en         : bit;
  signal bit_cnt_sclr       : bit;
  signal byte_cnt           : unsigned(2 downto 0); --Count 
  signal byte_cnt_en        : bit;
  signal byte_cnt_sclr      : bit;
  signal shft_reg           : unsigned(7 downto 0);
  signal shft_reg_ld_dat    : bit;
  signal shft_reg_ld_add_wr : bit;
  signal shft_reg_ld_add_rd : bit;
  signal shft_reg_ld_reg    : bit;
  signal shft_reg_left      : bit;
  signal sclda_ff           : unsigned(1 downto 0);
  signal sclda_ff_d         : unsigned(1 downto 0);
  signal ss1_run            : bit;
  signal ss1_mode           : unsigned(2 downto 0);
  signal ss1_cout           : bit;
--
-- State machine
--
  type ss1_type is (init,cnta,cntb,cntc,cntd);
  signal ss1_ps,ss1_ns : ss1_type;
  type ss2_type is (init,wr1,wr2,wr3,wr4,rd1,rd2,rd3,rd4,rd5,stop);
  signal ss2_ps,ss2_ns : ss2_type;
--
begin
--
-- define outputs
--
  reg_sel <= byte_cnt;
  dout <= shft_reg;
  scl_out <= sclda_ff(1);
  sda_out <= sclda_ff(0);
--
-- Clock processes, state machine and counter
--
  clk_proc : process(clk,ss1_ns,ss2_ns)
  begin
--  
    if (clk'event AND clk = '1') then 
      ss1_ps <= ss1_ns;
      ss2_ps <= ss2_ns;
      sclda_ff <= sclda_ff_d;
--
-- clock speed defined by port6 (bits 8,9)
-- 1MHz equals 961.5Khz (due to clock divider resolution)
-- 400Khz equals 390.6Khz (due to clock divider resolution)
-- 100Khz equals 100Khz
-- 
      if (clk_cnt_sclr = '1') OR
         ((clk_cnt = 12) AND (din2(9) = '1')) OR
         ((clk_cnt = 31) AND (din2(9 downto 8) ="01")) OR         
         ((clk_cnt = 124) AND (din2(9 downto 8) ="00")) then 
        clk_cnt_cout <= '1';
        clk_cnt <= "0000000000";
      else 
        clk_cnt <= clk_cnt + 1;
        clk_cnt_cout <= '0';
      end if;
--
      if (bit_cnt_sclr = '1') then bit_cnt <= "0000";  --bit counter for number of bits
      elsif (bit_cnt_en = '1') then bit_cnt <= bit_cnt + 1;
      else bit_cnt <= bit_cnt;
      end if;
--
      if (byte_cnt_sclr = '1') then byte_cnt <= "000";  --byte counter for number of bytes
      elsif (byte_cnt_en = '1') then byte_cnt <= byte_cnt + 1;
      else byte_cnt <= byte_cnt;
      end if;
--
      if shft_reg_left = '1' then shft_reg <= shft_reg(6 downto 0) & sda_in; --shift data in on read
      elsif shft_reg_ld_dat = '1' then
        case byte_cnt is
          when "000" => shft_reg <= din3(7 downto 0);
          when "001" => shft_reg <= din3(15 downto 8);
          when "010" => shft_reg <= din4(7 downto 0);
          when "011" => shft_reg <= din4(15 downto 8);
          when "100" => shft_reg <= din5(7 downto 0);
          when "101" => shft_reg <= din5(15 downto 8);
          when "110" => shft_reg <= din6(7 downto 0);
          when "111" => shft_reg <= din6(15 downto 8);
        end case;
      elsif shft_reg_ld_add_wr = '1' then shft_reg <= din1(6 downto 0)&'0';
      elsif shft_reg_ld_add_rd = '1' then shft_reg <= din1(6 downto 0)&'1';
      elsif shft_reg_ld_reg = '1' then shft_reg <= din1(15 downto 8);
      else shft_reg <= shft_reg;
      end if;
    end if;
  end process clk_proc;
--
-- Combinational process for state machine and output
--
  comb1_proc : process (ss1_ps,ss1_run,ss1_mode,clk_cnt_cout,shft_reg)
  begin
--
-- Two linked state machines, state machine 1 supports bit level commands
-- State machine 2 supports byte level commands
--
-- State machine 1 modes 000 to 111: idle, start, rep start, stop, write, read, ack, nack
--
--
  clk_cnt_sclr <= '0'; --default values
  ss1_cout <= '0';
--  
  case ss1_ps is 
    when init => --wait for run command from state machine 2
      clk_cnt_sclr <= '1';
      sclda_ff_d <= "11";
      if ss1_run = '1' then ss1_ns <= cnta;
      else ss1_ns <= init;
      end if;
    when cnta =>      
      case ss1_mode is 
        when "000" => sclda_ff_d <= "00"; --idle
        when "001" => sclda_ff_d <= "11"; --start
        when "010" => sclda_ff_d <= "01"; --rep start, mostly hold output high & stretch data/clk high
        when "011" => sclda_ff_d <= "00"; --stop
        when "100" => sclda_ff_d(1) <= '0'; sclda_ff_d(0) <= shft_reg(7); --write
        when "101" => sclda_ff_d <= "01"; --read, input to shift register is sca
        when "110" => sclda_ff_d <= "01"; --ack
        when "111" => sclda_ff_d <= "00"; --nack, low not used
      end case;
      if (clk_cnt_cout = '1') then ss1_ns <= cntb;
      else ss1_ns <= cnta;
      end if; 
    when cntb =>      
      case ss1_mode is 
        when "000" => sclda_ff_d <= "00"; --idle
        when "001" => sclda_ff_d <= "10"; --start
        when "010" => sclda_ff_d <= "11"; --rep start
        when "011" => sclda_ff_d <= "10"; --stop
        when "100" => sclda_ff_d(1) <= '1'; sclda_ff_d(0) <= shft_reg(7); --write
        when "101" => sclda_ff_d <= "11"; --read
        when "110" => sclda_ff_d <= "11"; --ack
        when "111" => sclda_ff_d <= "10"; --nack
      end case;
      if (clk_cnt_cout = '1') then ss1_ns <= cntc;
      else ss1_ns <= cntb;
      end if; 
    when cntc =>      
      case ss1_mode is 
        when "000" => sclda_ff_d <= "00"; --idle
        when "001" => sclda_ff_d <= "10"; --start
        when "010" => sclda_ff_d <= "11"; --rep start
        when "011" => sclda_ff_d <= "10"; --stop
        when "100" => sclda_ff_d(1) <= '1'; sclda_ff_d(0) <= shft_reg(7); --write
        when "101" => sclda_ff_d <= "11"; --read
        when "110" => sclda_ff_d <= "11"; --ack
        when "111" => sclda_ff_d <= "10"; --nack
      end case;
      if (clk_cnt_cout = '1') then ss1_ns <= cntd;
      else ss1_ns <= cntc;
      end if; 
    when cntd =>      
      case ss1_mode is 
        when "000" => sclda_ff_d <= "00"; --idle
        when "001" => sclda_ff_d <= "00"; --start
        when "010" => sclda_ff_d <= "11"; --rep start
        when "011" => sclda_ff_d <= "11"; --stop
        when "100" => sclda_ff_d(1) <= '0'; sclda_ff_d(0) <= shft_reg(7); --write
        when "101" => sclda_ff_d <= "01"; --read
        when "110" => sclda_ff_d <= "01"; --ack
        when "111" => sclda_ff_d <= "00"; --nack
      end case;
      if (clk_cnt_cout = '1') AND (ss1_run = '1') then 
        ss1_cout <= '1';
        ss1_ns <= cnta;
      elsif (clk_cnt_cout = '1') then 
        ss1_cout <= '1';
        ss1_ns <= init;
      else ss1_ns <= cntd;
      end if;
    when others => 
      sclda_ff_d <= "11";    
      ss1_ns <= init;
  end case;
  end process comb1_proc;
--
-- State machine 2 control machine 1 for read/write modes
--
  comb2_proc : process (ss1_ps,ss2_ps,bit_cnt,byte_cnt,shft_reg,sda_in,din2,ss1_cout,clk_cnt_cout)
  begin
--
  bit_cnt_sclr <= '0';  --bit count
  bit_cnt_en <= '0';
  byte_cnt_sclr <= '0'; --byte count
  byte_cnt_en <= '0';
  shft_reg_left <= '0'; --shift register
  shft_reg_ld_dat <= '0';
  shft_reg_ld_add_wr <= '0';
  shft_reg_ld_add_rd <= '0';
  shft_reg_ld_reg <= '0';
  reg_wr <= '0';        --write to port I/O, address is byte count
  clr_read <= '0';      --clear/set bits in port I/O
  clr_write <= '0';
  set_nack_err <= '0';
  ss1_run <= '0';		--control ss1
  ss1_mode <= "000";
--  
  case ss2_ps is
    when init =>
      bit_cnt_sclr <= '1';
      byte_cnt_sclr <= '1';
      if din2(0) = '1' then 
        shft_reg_ld_add_wr <= '1';
        ss2_ns <= wr1;
      elsif din2(1) = '1' then 
        shft_reg_ld_add_wr <= '1'; --read, first write to address
        ss2_ns <= rd1;
      else ss2_ns <= init;
      end if;
--
    when wr1 => --start, write slave address
      ss1_run <= '1';
      if (ss1_cout = '1') AND bit_cnt = "1001" then 
        bit_cnt_sclr <= '1';
        shft_reg_ld_reg <= '1';
        ss2_ns <= wr2;
      elsif (ss1_cout = '1') then
        ss2_ns <= wr1;
        bit_cnt_en <= '1';
        if bit_cnt /= "0000" then shft_reg_left <= '1';
        end if;
      elsif (ss1_ps = cntc) AND (clk_cnt_cout = '1') AND 
            (sda_in = '1') AND (bit_cnt = "1001") then --handle no ack
        clr_write <= '1';
        set_nack_err <= '1';
        ss2_ns <= stop;
      else ss2_ns <= wr1;
      end if;
      case bit_cnt is
        when "0000" => ss1_mode <= "001"; --start
        when "0001"|"0010"|"0011"|"0100" => ss1_mode <= "100"; --write;
        when "0101"|"0110"|"0111"|"1000" => ss1_mode <= "100"; --write;
        when "1001" => ss1_mode <= "110"; --ack;
        when others => ss1_mode <= "000";
      end case;
    when wr2 => --write register address
      ss1_run <= '1';
      if (ss1_cout = '1') AND bit_cnt = "1001" then 
        bit_cnt_sclr <= '1';
        shft_reg_ld_dat <= '1';
        ss2_ns <= wr3;
      elsif (ss1_cout = '1') then
        ss2_ns <= wr2;
        bit_cnt_en <= '1';
        if (bit_cnt /= "0000") then shft_reg_left <= '1';
        end if;
      elsif (ss1_ps = cntc) AND (clk_cnt_cout = '1') AND 
            (sda_in = '1') AND (bit_cnt = "1001") then --handle no ack
        clr_write <= '1';
        set_nack_err <= '1';
        ss2_ns <= stop;
      else ss2_ns <= wr2;
      end if;
      case bit_cnt is
        when "0000" => ss1_mode <= "000"; --idle
        when "0001"|"0010"|"0011"|"0100" => ss1_mode <= "100"; --write;
        when "0101"|"0110"|"0111"|"1000" => ss1_mode <= "100"; --write;
        when "1001" => ss1_mode <= "110"; --ack;
        when others => ss1_mode <= "000";
      end case;
    when wr3 => --write data 1 to 8 bytes
      ss1_run <= '1';
      if (ss1_cout = '1') AND (bit_cnt = "1001") AND byte_cnt = din2(4 downto 2) then
        clr_write <= '1'; 
        ss2_ns <= stop;
      elsif (ss1_cout = '1') AND bit_cnt = "1001" then 
        bit_cnt_sclr <= '1';
        byte_cnt_en <= '1';
        ss2_ns <= wr4;
      elsif (ss1_cout = '1') then
        bit_cnt_en <= '1';
        ss2_ns <= wr3;
        if (bit_cnt /= "0000") then shft_reg_left <= '1';
        end if;
      elsif (ss1_ps = cntc) AND (clk_cnt_cout = '1') AND 
            (sda_in = '1') AND (bit_cnt = "1001") then --handle no ack
        clr_write <= '1';
        set_nack_err <= '1';
        ss2_ns <= stop;
      else ss2_ns <= wr3;
      end if;
      case bit_cnt is
        when "0000" => ss1_mode <= "000"; --idle
        when "0001"|"0010"|"0011"|"0100" => ss1_mode <= "100"; --write;
        when "0101"|"0110"|"0111"|"1000" => ss1_mode <= "100"; --write;
        when "1001" => ss1_mode <= "110"; --ack;
        when others => ss1_mode <= "000"; 
      end case;
    when wr4 => --used to load data, need one clock delay for byte count update
    shft_reg_ld_dat <= '1';
      ss1_run <= '1';
      case bit_cnt is
        when "0000" => ss1_mode <= "000"; --idle
        when "0001"|"0010"|"0011"|"0100" => ss1_mode <= "100"; --write;
        when "0101"|"0110"|"0111"|"1000" => ss1_mode <= "100"; --write;
        when "1001" => ss1_mode <= "110"; --ack;
        when others => ss1_mode <= "000"; 
      end case;
      ss2_ns <= wr3;
--    
    when rd1 => --load slave address
      ss1_run <= '1';
      if (ss1_cout = '1') AND bit_cnt = "1001" then 
        bit_cnt_sclr <= '1';
        shft_reg_ld_reg <= '1';
        ss2_ns <= rd2;
      elsif (ss1_cout = '1') then
        ss2_ns <= rd1;
        bit_cnt_en <= '1';
        if bit_cnt /= "0000" then shft_reg_left <= '1';
        end if;
      elsif (ss1_ps = cntc) AND (clk_cnt_cout = '1') AND 
            (sda_in = '1') AND (bit_cnt = "1001") then --handle no ack
        clr_read <= '1';
        set_nack_err <= '1';
        ss2_ns <= stop;
      else ss2_ns <= rd1;
      end if;
      case bit_cnt is
        when "0000" => ss1_mode <= "001"; --start
        when "0001"|"0010"|"0011"|"0100" => ss1_mode <= "100"; --write;
        when "0101"|"0110"|"0111"|"1000" => ss1_mode <= "100"; --write;
        when "1001" => ss1_mode <= "110"; --ack;
        when others => ss1_mode <= "000";
      end case;    
    when rd2 => --load register address
      ss1_run <= '1';
      if (ss1_cout = '1') AND bit_cnt = "1001" then 
        bit_cnt_sclr <= '1';
        ss2_ns <= rd3;
      elsif (ss1_cout = '1') then
        ss2_ns <= rd2;
        bit_cnt_en <= '1';
        if bit_cnt /= "0000" then shft_reg_left <= '1';
        end if;
      elsif (ss1_ps = cntc) AND (clk_cnt_cout = '1') AND 
            (sda_in = '1') AND (bit_cnt = "1001") then --handle no ack
        clr_read <= '1';
        set_nack_err <= '1';
        ss2_ns <= stop;
      else ss2_ns <= rd2;
      end if;
      case bit_cnt is
        when "0000" => ss1_mode <= "000"; --idle
        when "0001"|"0010"|"0011"|"0100" => ss1_mode <= "100"; --write;
        when "0101"|"0110"|"0111"|"1000" => ss1_mode <= "100"; --write;
        when "1001" => ss1_mode <= "110"; --ack;
        when others => ss1_mode <= "000";
      end case;
    when rd3 => --hold clk/data high for one clock cycle
      ss1_run <= '1';
      ss1_mode <= "010"; --repeated start
      if (ss1_cout = '1') then 
        shft_reg_ld_add_rd <= '1';        
        ss2_ns <= rd4;
      else ss2_ns <= rd3;
      end if;
    when rd4 => --load slave address
      ss1_run <= '1';
      if (ss1_cout = '1') AND bit_cnt = "1001" then 
        bit_cnt_sclr <= '1';
        ss2_ns <= rd5;
      elsif (ss1_cout = '1') then
        ss2_ns <= rd4;
        bit_cnt_en <= '1';
        if bit_cnt /= "0000" then shft_reg_left <= '1';
        end if;
      elsif (ss1_ps = cntc) AND (clk_cnt_cout = '1') AND 
            (sda_in = '1') AND (bit_cnt = "1001") then --handle no ack
        clr_read <= '1';        
        set_nack_err <= '1';
        ss2_ns <= stop;
      else ss2_ns <= rd4;
      end if;
      case bit_cnt is
        when "0000" => ss1_mode <= "001"; --start
        when "0001"|"0010"|"0011"|"0100" => ss1_mode <= "100"; --write;
        when "0101"|"0110"|"0111"|"1000" => ss1_mode <= "100"; --write;
        when "1001" => ss1_mode <= "110"; --ack;
        when others => ss1_mode <= "000";
      end case;
    when rd5 => --read slave data
      ss1_run <= '1';
      if (ss1_cout = '1') AND bit_cnt = "1001" AND byte_cnt = din2(4 downto 2) then
        clr_read <= '1'; 
        reg_wr <= '1'; -- write to port registers
        ss2_ns <= stop;
      elsif (ss1_cout = '1') AND bit_cnt = "1001" then 
        bit_cnt_sclr <= '1';
        byte_cnt_en <= '1';
        reg_wr <= '1'; -- write to port registers
        ss2_ns <= rd5;
      elsif (ss1_cout = '1') then
        bit_cnt_en <= '1';
        ss2_ns <= rd5;
      elsif (clk_cnt_cout = '1') AND (ss1_ps = cntc) AND 
            (bit_cnt /= "0000") AND (bit_cnt /= "1001") then  --latch data at falling edge of scl
        shft_reg_left <= '1';
        ss2_ns <= rd5;
      else ss2_ns <= rd5;
      end if;
      case bit_cnt is
        when "0000" => ss1_mode <= "000"; --idle
        when "0001"|"0010"|"0011"|"0100" => ss1_mode <= "101"; --read;
        when "0101"|"0110"|"0111"|"1000" => ss1_mode <= "101"; --read;
        when "1001" => 
          if byte_cnt = din2(4 downto 2) then ss1_mode <= "110"; --ack;
          else ss1_mode <= "111"; --nack
          end if;
        when others => ss1_mode <= "000"; 
      end case;
--
    when stop =>
      ss1_mode <= "011";
      if ss1_cout = '1' then ss2_ns <= init;
      else
        ss1_run <= '1';
        ss2_ns <= stop;
      end if;
    when others => ss2_ns <= init;
--
  end case; 
  end process comb2_proc;
end;
