--
--Outputs byte to serial port. Assumes din is held valid during the
--output serial stream. The byte sent line is asserted one clock at end of
--transmission of data.
--
--Ron Prusia
--March 2024 - Original version
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
entity comm_uart_tx is
port (
  clk	      : in std_logic;   --clock
  num_bits    : in std_logic;   --'0': 8 bits, '1': 7 bits
  num_stop    : in std_logic;   --'0': 1 stop bit, '1': 2 stop bits
  parity_en   : in std_logic;   --'0': parity not enabled, '1': partity enabled
  par_mode    : in unsigned(1 downto 0);  --'00': partity 0, '01': parity 1, '10': parity even, '11': parity odd
  baud_divide : in unsigned(15 downto 0); --baud rate divide count
  data_rdy    : in std_logic;             --Flag to indicate there is a byte to transmit
  data_in     : in unsigned(7 downto 0);  --Data to transmit
--
  tx          : out std_logic; --Output transmitted data
  byte_sent   : out bit        --Flag to indicate the byte was sent
  );
end comm_uart_tx;
--
architecture rtl of comm_uart_tx is
--
-- Variables
--
  signal shft_reg       : unsigned(7 downto 0); --includes start stop bits
  signal shft_load      : std_logic;
  signal shft_right     : std_logic;            --Shift input data
  signal tx_ff          : std_logic := '1';     --Output register, default high
  signal tx_ff_d        : std_logic;
  signal bit_cnt        : unsigned(2 downto 0);	--Output from mux to output register
  signal bit_cnt_sclr   : std_logic;            --Sync clear
  signal bit_cnt_en     : std_logic;            --Count enable
  signal baud_cnt       : unsigned(15 downto 0);--Baud rate counter
  signal baud_cnt_sclr  : std_logic;            --Sync clear
  signal baud_cnt_cout  : std_logic;            --Match of counter to baud rate input
  signal parity_cnt     : unsigned(2 downto 0);
  signal parity_cnt_en  : std_logic;
  signal parity_cnt_sclr: std_logic;
  
--
  type ss_type is (init,start,bits_out,parity,stop1,stop2);
  signal ss_ps,ss_ns : ss_type;
--
  begin
  tx <= tx_ff;
--
-- Clock processes
--
  clk_proc : process(clk,ss_ns,baud_cnt_sclr,bit_cnt_sclr,data_rdy,data_in,shft_load,shft_right,
                      parity_cnt,parity_cnt_sclr)
  begin
    if (clk'event AND clk = '1') then 
      ss_ps <= ss_ns;
      tx_ff <= tx_ff_d;
--
      if (shft_load = '1') then shft_reg <= data_in; --latch input byte on data_in ready, lsb out first
      elsif (shft_right = '1') then shft_reg <= '1' & shft_reg(7 downto 1); --shift right
      else shft_reg <= shft_reg;
      end if;      
--
      if (parity_cnt_sclr = '1') then parity_cnt <= "000";
      elsif (parity_cnt_en = '1') then parity_cnt <= parity_cnt + 1; 
      else parity_cnt <= parity_cnt;
      end if;
--
      if (baud_cnt_sclr = '1') then
        baud_cnt_cout <= '0';
        baud_cnt <= x"0000";
      elsif (baud_cnt = baud_divide) then 
        baud_cnt_cout <= '1';
        baud_cnt <= x"0000";
      else 
        baud_cnt <= baud_cnt + 1;
        baud_cnt_cout <= '0';
      end if;
--
      if (bit_cnt_sclr = '1') then bit_cnt <= "000";
      elsif (bit_cnt_en = '1') then bit_cnt <= bit_cnt + 1;
      else bit_cnt <= bit_cnt;
      end if;      
      
    end if;
  end process clk_proc;
--
-- The state machine waits for the data ready signal it then proceeds to output
-- the bits. Start bit is first. The output is high, until output started. 
--
  ss_comb_proc : process(ss_ps,ss_ns,data_rdy,bit_cnt,baud_cnt_cout,shft_reg,num_bits,
                         parity_en,par_mode,parity_cnt,num_stop)
  begin
--
    bit_cnt_sclr <= '0'; --default state for variables is not asserted
    bit_cnt_en <= '0';
    baud_cnt_sclr <= '0';
    byte_sent <= '0';
    shft_right <= '0';
    shft_load <= '0';
    parity_cnt_en <= '0';
    parity_cnt_sclr <= '0';
    tx_ff_d <= '1';      --default should be TX output high
--
    case ss_ps is
      when init => --Wait the ready signal
        bit_cnt_sclr <= '1';
        baud_cnt_sclr <= '1';
        parity_cnt_sclr <= '1';
        if (data_rdy = '1') then 
          shft_load <= '1';
          ss_ns <= start;
        else ss_ns <= init;
        end if;
      when start =>  --Output start bit, low for one baud cycle
        tx_ff_d <= '0';
        if (baud_cnt_cout = '1') then ss_ns <= bits_out;
        else ss_ns <= start;
        end if;
      when bits_out =>
        tx_ff_d <= shft_reg(0);
        if ((baud_cnt_cout = '1') AND (bit_cnt = "110") AND (num_bits = '1')) OR   --seven bits check
           ((baud_cnt_cout = '1') AND (bit_cnt = "111") AND (num_bits = '0')) then --eight bits
          if shft_reg(0) = '1' then parity_cnt_en <= '1';  --count parity
          end if;
          if (parity_en = '1') then ss_ns  <= parity;
          else ss_ns <= stop1;
          end if;
        elsif (baud_cnt_cout = '1') then --shift data
          if shft_reg(0) = '1' then parity_cnt_en <= '1';  --count parity
          end if;
          shft_right <= '1';
          bit_cnt_en <= '1';
          ss_ns <= bits_out;
        else ss_ns <= bits_out;
        end if;
      when parity =>
        if (baud_cnt_cout = '1') then ss_ns <= stop1;
        else ss_ns <= parity;
        end if;     
        case par_mode is      
	      when "00" => tx_ff_d <= '0';
	      when "01" => tx_ff_d <= '1';
	      when "10"|"11" => tx_ff_d <= parity_cnt(0); 
	    end case;
      when stop1 => --first stop bit, default output is asserted
        if (baud_cnt_cout = '1') AND (num_stop = '1') then ss_ns <= stop2;
        elsif (baud_cnt_cout = '1') then 
          byte_sent <= '1';
          ss_ns <= init;
        else ss_ns <= stop1;
        end if;
      when stop2 => --second stop bit, default output is asserted
        if (baud_cnt_cout = '1') then
          byte_sent <= '1';
          ss_ns <= init;
        else ss_ns <= stop2;
        end if;
      when others => ss_ns <= init;
	end case;
  end process ss_comb_proc;
end;
