--
--Outputs byte to serial port. Assumes din is held valid during the
--output serial stream. The byte sent line is asserted one clock at end of
--transmission of data.
--
--Ron Prusia
--March 2024 - Original version
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
entity comm_uart_rx is
port (
  clk	      : in std_logic;   --clock
  rx_in       : in std_logic;   --input serial data
  num_bits    : in std_logic;   --'0': 8 bits, '1': 7 bits
  num_stop    : in std_logic;   --'0': 1 stop bit, '1': 2 stop bits
  par_en      : in std_logic;   --'0': parity not enabled, '1': partity enabled
  par_mode    : in unsigned(1 downto 0);  --'00': partity 0, '01': parity 1, '10': parity even, '11': parity odd
  baud_divide : in unsigned(15 downto 0); --baud rate divide count
--
  frame_err   : out std_logic;
  parity_err  : out std_logic; 
  dout        : out unsigned(7 downto 0); --Output transmitted data
  byte_rdy    : out std_logic             --Flag to indicate byte is ready
  );
end comm_uart_rx;
--
architecture rtl of comm_uart_rx is
--
-- Variables
--
  signal sync_ff        : unsigned(2 downto 0):= "111"; --sync registers for input serial port data
  signal shft_reg       : unsigned(7 downto 0); --includes start stop bits
  signal shft_right     : std_logic;            --Shift input data
  signal shft_right7    : std_logic;    
  signal bit_cnt        : unsigned(2 downto 0);	--Output from mux to output register
  signal bit_cnt_sclr   : std_logic;            --Sync clear
  signal bit_cnt_en     : std_logic;            --Count enable
  signal baud_cnt       : unsigned(15 downto 0);--Baud rate counter
  signal baud_cnt_sclr  : std_logic;            --Sync clear
  signal baud_cnt_cout  : std_logic;            --Match of counter to baud rate input
  signal parity_cnt     : unsigned(2 downto 0);
  signal parity_cnt_en  : std_logic;
  signal parity_cnt_sclr: std_logic;
  
--
  type ss_type is (init,start,bits_in,parity1,parity2,stop1,stop2);
  signal ss_ps,ss_ns : ss_type;
--
  begin
--
-- Outputs
--
  dout <= shft_reg;
--
-- Clock processes
--
  clk_proc : process(clk,ss_ns,baud_cnt_sclr,bit_cnt_sclr,rx_in,shft_right,parity_cnt,parity_cnt_sclr)
  begin
    if (clk'event AND clk = '1') then 
      ss_ps <= ss_ns;
      sync_ff <= sync_ff(1 downto 0) & rx_in;
--
      if (shft_right = '1') then shft_reg <= sync_ff(2) & shft_reg(7 downto 1); --shift right
      elsif (shft_right7 = '1') then shft_reg <= '0' & sync_ff(2) & shft_reg(7 downto 2); --double shift with zero fill for last shift 7 bits
      else shft_reg <= shft_reg;
      end if;      
--
      if (parity_cnt_sclr = '1') then parity_cnt <= "000";
      elsif (parity_cnt_en = '1') then parity_cnt <= parity_cnt + 1; 
      else parity_cnt <= parity_cnt;
      end if;
--
      if (baud_cnt_sclr = '1') then
        baud_cnt_cout <= '0';
        baud_cnt <= x"0000";
      elsif (baud_cnt = baud_divide) then 
        baud_cnt_cout <= '1';
        baud_cnt <= x"0000";
      else 
        baud_cnt <= baud_cnt + 1;
        baud_cnt_cout <= '0';
      end if;
--
      if (bit_cnt_sclr = '1') then bit_cnt <= "000";
      elsif (bit_cnt_en = '1') then bit_cnt <= bit_cnt + 1;
      else bit_cnt <= bit_cnt;
      end if;      
      
    end if;
  end process clk_proc;
--
-- The state machine waits for the data ready signal it then proceeds to output
-- the bits. Start bit is first. The output is high, until output started. 
--
  ss_comb_proc : process(ss_ps,ss_ns,bit_cnt,baud_cnt_cout,shft_reg,num_bits,sync_ff,baud_cnt,baud_divide,
                         par_en,par_mode,parity_cnt,num_stop)
  begin
--
    bit_cnt_sclr <= '0'; --default state for variables is not asserted
    bit_cnt_en <= '0';
    baud_cnt_sclr <= '0';
    byte_rdy <= '0';
    shft_right <= '0';
    shft_right7 <= '0';
    parity_cnt_en <= '0';
    parity_cnt_sclr <= '0';
    parity_err <= '0';
    frame_err <= '0';
--
    case ss_ps is
      when init => --Wait the ready signal
        bit_cnt_sclr <= '1';
        baud_cnt_sclr <= '1';
        parity_cnt_sclr <= '1';
        if (sync_ff(2) = '0') then ss_ns <= start; --data starts with low input
        else ss_ns <= init;
        end if;
      when start =>  --Wait half baud count for start bit
        if (baud_cnt(14 downto 0) = baud_divide(15 downto 1)) then 
          baud_cnt_sclr <= '1'; -- restart baud count
          ss_ns <= bits_in; --half baud count
        else ss_ns <= start;
        end if;
      when bits_in =>
        if (baud_cnt_cout = '1') AND (bit_cnt = "111") AND (num_bits = '0') then --eight bits
          shft_right <= '1';
          if sync_ff(2) = '1' then parity_cnt_en <= '1';  --count parity
          end if;
          if (par_en = '1') then ss_ns  <= parity1;
          else ss_ns <= stop1;
          end if;
        elsif (baud_cnt_cout = '1') AND (bit_cnt = "110") AND (num_bits = '1') then   --seven bits check
          shft_right7 <= '1'; --special shift for last bit of seven, zero fill with double shift
          if sync_ff(2) = '1' then parity_cnt_en <= '1';  --count parity
          end if;
          if (par_en = '1') then ss_ns  <= parity1;
          else ss_ns <= stop1;
          end if;
        elsif (baud_cnt_cout = '1') then --shift data
          if sync_ff(2) = '1' then parity_cnt_en <= '1';  --count parity
          end if;
          shft_right <= '1';
          bit_cnt_en <= '1';
          ss_ns <= bits_in;
        else ss_ns <= bits_in;
        end if;
      when parity1 => --count parity bit
        if (baud_cnt_cout = '1') then 
          if sync_ff(2) = '1' then parity_cnt_en <= '1';  --count parity
          end if;          
          ss_ns <= parity2;
        else ss_ns <= parity1;
        end if;
      when parity2 =>
        case par_mode is
          when "00" => if sync_ff(2) = '1' then parity_err <= '1'; end if;
          when "01" => if sync_ff(2) = '0' then parity_err <= '1'; end if;
          when "10"|"11" => if parity_cnt(0) /= par_mode(0) then parity_err <= '1'; end if;
        end case;          
        ss_ns <= stop1;
      when stop1 => --first stop bit, default output is asserted
        if (baud_cnt_cout = '1') AND (num_stop = '1') then 
          if sync_ff(2) = '0' then frame_err <= '1';
          end if;
          ss_ns <= stop2;
        elsif (baud_cnt_cout = '1') then 
          byte_rdy <= '1';
          if sync_ff(2) = '0' then frame_err <= '1';
          end if;
          ss_ns <= init;
        else ss_ns <= stop1;
        end if;
      when stop2 => --second stop bit, default output is asserted
        if (baud_cnt_cout = '1') then
          byte_rdy <= '1';
          if sync_ff(2) = '0' then frame_err <= '1';
          end if;
          ss_ns <= init;
        else ss_ns <= stop2;
        end if;
      when others => ss_ns <= init;
	end case;
  end process ss_comb_proc;
end;
