--
--   Copyright (C) 2023 Ron Prusia
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program.  If not, see <http://www.gnu.org/licenses.
--
--  Controller for seven segment multiplexing, up to eight digits
--
-- June, 2023 baseline
--         
-- Ron Prusia
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
entity seg_ctrl is
port (
  clk           : in bit;					--Clock input
  port1         : in unsigned(15 downto 0); --segment control
  port2         : in unsigned(15 downto 0); --segments 0-3
  port3         : in unsigned(15 downto 0); --segments 4-7
--
  a,b,c,d,e,f,g : out bit;
  dp            : out bit;
  seg_sel       : out unsigned(7 downto 0)
  );
end seg_ctrl;
--
architecture rtl of seg_ctrl is
--
-- Variables
--
  signal clk_cnt        : unsigned(13 downto 0); --divider multiplexer (4Khz)
  signal seg_cnt        : unsigned(2 downto 0);
  signal seg_sel_ff     : unsigned(7 downto 0); --register outputs
  signal seg_sel_ff_d1  : unsigned(7 downto 0); --allows two assignments for inversion
  signal seg_sel_ff_d2  : unsigned(7 downto 0);
  signal seg_ff         : unsigned(6 downto 0);
  signal seg_ff_d1      : unsigned(6 downto 0); --allows two assignments for inversion
  signal seg_ff_d2      : unsigned(6 downto 0);
  signal seg_nibble     : unsigned(3 downto 0);
  signal dp_ff          : bit;
  signal dp_ff_d1       : bit;
  signal dp_ff_d2       : bit;
--
-- State machine
--
  type ss_type is (init,sync,dataout);
  signal ss_ps,ss_ns : ss_type;
--  
begin
  seg_sel <= seg_sel_ff;
  a <= to_bit(seg_ff(0));
  b <= to_bit(seg_ff(1));
  c <= to_bit(seg_ff(2));
  d <= to_bit(seg_ff(3));
  e <= to_bit(seg_ff(4));
  f <= to_bit(seg_ff(5));
  g <= to_bit(seg_ff(6));
  dp <= dp_ff;
--
-- Clock processes, state machine and counter
--
  clk_proc : process(clk)
  begin
--  
    if (clk'event) AND (clk = '1') then 
      seg_sel_ff <= seg_sel_ff_d2;
      seg_ff <= seg_ff_d2;
      dp_ff <= dp_ff_d2;
--    
      if (clk_cnt = "11000011010011") then --clk divider for 4khz, sim with count of 3, "11000011010011"
         clk_cnt <= "00000000000000";
         if seg_cnt = port3(14 downto 12) then seg_cnt <= "000";--max segment count
         else seg_cnt <= seg_cnt + 1;
         end if;
      else clk_cnt <=  clk_cnt + 1;
      end if;
    end if;
  end process clk_proc;
--
-- Combinational process
--
  comb_proc : process(port1,port2,port3,seg_cnt,seg_sel_ff_d1,seg_nibble,seg_ff_d1,dp_ff_d1)
  begin
--
-- generate output logic
--
--
-- select segment enable, and which nibble to output
--
  case seg_cnt is
    when "000" => 
      seg_sel_ff_d1 <= x"01";
      seg_nibble <= port1(3 downto 0);
      dp_ff_d1 <= to_bit(port3(0));
    when "001" =>
      seg_sel_ff_d1 <= x"02";
      seg_nibble <= port1(7 downto 4);
      dp_ff_d1 <= to_bit(port3(1));
    when "010" =>
      seg_sel_ff_d1 <= x"04";
      seg_nibble <= port1(11 downto 8);
      dp_ff_d1 <= to_bit(port3(2));
    when "011" =>
      seg_sel_ff_d1 <= x"08"; 
      seg_nibble <= port1(15 downto 12);
      dp_ff_d1 <= to_bit(port3(3));
    when "100" => 
      seg_sel_ff_d1 <= x"10"; 
      seg_nibble <= port2(3 downto 0);
      dp_ff_d1 <= to_bit(port3(4));
    when "101" => 
      seg_sel_ff_d1 <= x"20"; 
      seg_nibble <= port2(7 downto 4);
      dp_ff_d1 <= to_bit(port3(5));
    when "110" => 
      seg_sel_ff_d1 <= x"40"; 
      seg_nibble <= port2(11 downto 8);
      dp_ff_d1 <= to_bit(port3(6));
    when "111" => 
      seg_sel_ff_d1 <= x"80"; 
      seg_nibble <= port2(15 downto 12);
      dp_ff_d1 <= to_bit(port3(7));
  end case;
  if (port3(9) = '0') then seg_sel_ff_d2 <= seg_sel_ff_d1;
  else seg_sel_ff_d2 <= NOT (seg_sel_ff_d1); --invert
  end if;
--
-- bcd to segment decoder, segment 'a' is lsb, segment 'g' is msb
--
  case seg_nibble is
    when "0000" => seg_ff_d1 <= "0111111"; --0
    when "0001" => seg_ff_d1 <= "0000110"; --1
    when "0010" => seg_ff_d1 <= "1011011"; --2
    when "0011" => seg_ff_d1 <= "1001111"; --3
    when "0100" => seg_ff_d1 <= "1100110"; --4
    when "0101" => seg_ff_d1 <= "1101101"; --5
    when "0110" => seg_ff_d1 <= "1111101"; --6
    when "0111" => seg_ff_d1 <= "0000111"; --7
    when "1000" => seg_ff_d1 <= "1111111"; --8
    when "1001" => seg_ff_d1 <= "1101111"; --9
    when "1010" => seg_ff_d1 <= "1110111"; --a
    when "1011" => seg_ff_d1 <= "1111100"; --b
    when "1100" => seg_ff_d1 <= "0111001"; --c
    when "1101" => seg_ff_d1 <= "1011110"; --d
    when "1110" => seg_ff_d1 <= "1111001"; --e
    when "1111" => seg_ff_d1 <= "1110001"; --f
  end case;
  if (port3(8) = '0') then 
    seg_ff_d2 <= seg_ff_d1;
    dp_ff_d2 <= dp_ff_d1;
  else 
    seg_ff_d2 <= NOT (seg_ff_d1); --invert
    dp_ff_d2 <= NOT (dp_ff_d1);
  end if;
--
  end process comb_proc;  
end;
