--
--   Copyright (C) 2023 Ron Prusia
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program.  If not, see <http://www.gnu.org/licenses.
--
--  Controller for seven segment multiplexing, up to eight digits
--
-- June, 2023 baseline
-- Supports either one 64x32 panel or four 32x16 panels
-- Supports FM6047 with registers 2 and 3 set to 0x7FFF and 0x0040
--                 also latch enable is asserted last three clocks of write cycle
--         
-- Ron Prusia
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
--
entity led_matrix is
generic (scan_8to1 : std_logic := '0');
port (
  clk         : in std_logic;               --Clock input
  mem1,mem2   : in unsigned(23 downto 0);   --Input memory
--
  mem_raddr   : out unsigned(9 downto 0);   --Read address for memory 
  r1,r2       : out std_logic;				--Pixel data out red
  g1,g2       : out std_logic;				--Pixel data out green
  b1,b2       : out std_logic;				--Pixel data out blue
  mat_clk     : out std_logic;				--Clock 
  a,b,c,d     : out std_logic;              --Line select
  mat_oe_n    : out std_logic;
  mat_le      : out std_logic
  );
end led_matrix;
--
architecture rtl of led_matrix is
--
-- Variables
--
  signal clk_cnt        : unsigned(1 downto 0);            --clock divider, four states per clock (6.25Mhz)
  signal clk_cnt_cout   : std_logic;
  signal pixel_cnt      : unsigned(7 downto 0);
  signal pixel_cnt_en   : std_logic;
  signal pixel_cnt_sclr : std_logic;
  signal bit_cnt        : unsigned(2 downto 0);
  signal bit_cnt_en     : std_logic;
  signal bit_cnt_sclr   : std_logic;
  signal line_cnt       : unsigned(3 downto 0);
  signal line_cnt_en    : std_logic;
  signal line_cnt_sclr  : std_logic;
  signal delay_cnt      : unsigned(14 downto 0);
  signal delay_cnt_sclr : std_logic;
  signal delay_flag     : std_logic;
  signal out_ff         : std_logic_vector(12 downto 0); --used to register all outputs
  signal out_ff_d       : std_logic_vector(12 downto 0); --"b2,b1,g2,g1,r2,r1,d,c,b,a,mat_stb,mat_oe,mat_clk"
  signal max_pixel      : unsigned(7 downto 0);
  signal max_line       : unsigned(3 downto 0);
  signal reg_set        : std_logic;
  signal reg_clr        : std_logic;
--
-- State machine
--
  type ss_type is (init,reg2a,reg2b,reg3a,reg3b,dis_a,dis_b,delaya,delayb);
  signal ss_ps,ss_ns : ss_type;
--  
begin
--
-- define outputs
--
  mat_clk <= out_ff(0); --mapping for output flip flop
  mat_oe_n <= NOT out_ff(1);
  mat_le <= out_ff(2); --asserted low, invert at output
  a <= out_ff(3);
  b <= out_ff(4);
  c <= out_ff(5);
  d <= out_ff(6);
  r1 <= out_ff(7);
  r2 <= out_ff(8);
  g1 <= out_ff(9);
  g2 <= out_ff(10);
  b1 <= out_ff(11);
  b2 <= out_ff(12);
--
-- define max values for output type, add memory addressses
--
  max_proc : process(all)
  begin
    if (scan_8to1 = '1') then
      max_pixel <= x"80"; --max pixel 128
      max_line <= "0111";
      mem_raddr <= line_cnt(2 downto 0) & pixel_cnt(6 downto 0);
    else
      max_pixel <= x"40"; --max pixel 64
      max_line <= "1111";
      mem_raddr <= line_cnt(3 downto 0) & pixel_cnt(5 downto 0);
    end if;
  end process max_proc;
--
-- Clock processes, state machine and counter
--
  clk_proc : process(clk)
  begin
--  
    if (clk'event) AND (clk = '1') then
      out_ff <= out_ff_d;
      ss_ps <= ss_ns;
--    
      if (clk_cnt = "11") then
        clk_cnt_cout <= '1';
        clk_cnt <= "00";
      else 
        clk_cnt <= clk_cnt + 1;
        clk_cnt_cout <= '0';
      end if;
--
      if (pixel_cnt_sclr = '1') then pixel_cnt <= x"00"; --max 128 pixels
      elsif (pixel_cnt_en = '1') then pixel_cnt <= pixel_cnt + 1;
      else pixel_cnt <= pixel_cnt;
      end if;
--
      if (bit_cnt_sclr = '1') then bit_cnt <= "000"; --max 8 bits
      elsif (bit_cnt_en = '1') then bit_cnt <= bit_cnt + 1;
      else bit_cnt <= bit_cnt;
      end if;
--
      if (line_cnt_sclr = '1') then line_cnt <= "0000"; --max 16 lines
      elsif (line_cnt_en = '1') then line_cnt <= line_cnt + 1;
      else line_cnt <= line_cnt;
      end if;
--
      if (delay_cnt_sclr = '1') then delay_cnt <= "000000000000000";
      else delay_cnt <= delay_cnt + 1;
      end if;
    end if;
  end process clk_proc;
--
-- Combinational process
-- generate output bits from memory or for setting registers, and chose delay counter bit
--
  comb_proc : process(ss_ps,mem1,mem2,clk_cnt_cout,delay_cnt,pixel_cnt,bit_cnt,line_cnt,max_line,max_pixel,clk_cnt,delay_flag,reg_set,reg_clr)
  begin
--
  pixel_cnt_sclr <= '0'; --default control values
  pixel_cnt_en <= '0';
  bit_cnt_sclr <= '0';
  bit_cnt_en <= '0';
  line_cnt_sclr <= '0';
  line_cnt_en <= '0';
  delay_cnt_sclr <= '0';
  delay_flag <= '0';
  reg_clr <= '0';
  reg_set <= '0';
  out_ff_d(6 downto 3) <= std_logic_vector(line_cnt); --sorry output ff is a bit confusing
  out_ff_d(2 downto 0) <= "000"; --2:le, 1:oe, 0:clk  
--
-- state machine to set registers, and then run output
--
  case ss_ps is
    when init =>
      pixel_cnt_sclr <= '1';
      if (clk_cnt_cout = '1') then ss_ns <= reg2a; --use clk_cnt to control speed of state machine
      else ss_ns <= init;
      end if;
    when reg2a => --write to register 2 (0x7FFF), clock low
      if pixel_cnt(3 downto 0) = "1111" then reg_clr <= '1'; --set output data
      else reg_set <= '1';
      end if;
      if (pixel_cnt > max_pixel-13) then out_ff_d(2) <= '1'; --latch enable high for 12 clocks to latch in data in register 2
      end if;
      if (clk_cnt_cout = '1') AND (pixel_cnt = max_pixel) then
        pixel_cnt_sclr <= '1';
        ss_ns <= reg3a;
      elsif (clk_cnt_cout = '1') then ss_ns <= reg2b;
      else ss_ns <= reg2a;
      end if;
    when reg2b => --clock high
      out_ff_d(0) <= '1'; --clock high
      if pixel_cnt(3 downto 0) = "1111" then reg_clr <= '1'; --set output data
      else reg_set <= '1';
      end if;
      if (pixel_cnt > max_pixel-13) then out_ff_d(2) <= '1'; --latch enable high for 12 clocks to latch in data in register 2
      end if;      
      if (clk_cnt_cout = '1') then
        pixel_cnt_en <= '1';
        ss_ns <= reg2a;
      else ss_ns <= reg2b;
      end if;    
    when reg3a => --write to register 3 (0x0040), clock low
      if pixel_cnt(3 downto 0) = "0110" then reg_set <= '1'; --set output data
      else reg_clr <= '1';
      end if;
      if (pixel_cnt > max_pixel-14) then out_ff_d(2) <= '1'; --latch enable high for 13 clocks to latch in data in register 3
      end if;
      if (clk_cnt_cout = '1') AND (pixel_cnt = max_pixel) then
        pixel_cnt_sclr <= '1';
        bit_cnt_sclr <= '1';
        line_cnt_sclr <= '1';
        ss_ns <= dis_a;
      elsif (clk_cnt_cout = '1') then ss_ns <= reg3b;
      else ss_ns <= reg3a;
      end if;
    when reg3b => --clock high
      out_ff_d(0) <= '1'; --clock high
      if pixel_cnt(3 downto 0) = "0110" then reg_set <= '1'; --set output data
      else reg_clr <= '1';
      end if;
      if (pixel_cnt > max_pixel-14) then out_ff_d(2) <= '1'; --latch enable high for 13 clocks to latch in data in register 3
      end if;      
      if (clk_cnt_cout = '1') then
        pixel_cnt_en <= '1';
        ss_ns <= reg3a;
      else ss_ns <= reg3b;
      end if; 
    when dis_a => --clock low
      if (pixel_cnt > max_pixel - 4) then out_ff_d(2) <= '1'; --latch enable high for 3 clocks to latch in data for FM6047
      end if;
      if (clk_cnt_cout = '1') AND (pixel_cnt = max_pixel) then
        pixel_cnt_sclr <= '1';
        delay_cnt_sclr <= '1';
        ss_ns <= delaya;
      elsif (clk_cnt_cout = '1') then ss_ns <= dis_b;
      else ss_ns <= dis_a;
      end if;
    when dis_b => --clock high
      out_ff_d(0) <= '1'; --clock high
      if (pixel_cnt > max_pixel - 4) then out_ff_d(2) <= '1'; --latch enable high for 3 clocks to latch in data for FM6047
      end if;
      if (clk_cnt_cout = '1') then
        pixel_cnt_en <= '1';
        delay_cnt_sclr <= '1';
        ss_ns <= dis_a;
      else ss_ns <= dis_b;
      end if;
    when delaya => --delay according to bit position
      out_ff_d(1) <= '1';
      case bit_cnt is
        when "000" => if delay_cnt > 127 then delay_flag <= '1'; end if; --127
        when "001" => if delay_cnt > 255 then delay_flag <= '1'; end if;
        when "010" => if delay_cnt > 511 then delay_flag <= '1'; end if;
        when "011" => if delay_cnt > 1023 then delay_flag <= '1'; end if;
        when "100" => if delay_cnt > 2047 then delay_flag <= '1'; end if;
        when "101" => if delay_cnt > 4095 then delay_flag <= '1'; end if;
        when "110" => if delay_cnt > 8191 then delay_flag <= '1'; end if;
        when "111" => if delay_cnt > 16383 then delay_flag <= '1'; end if;
      end case;
      if (delay_flag = '1') then ss_ns <= delayb;
      else ss_ns <= delaya;
      end if;
    when delayb =>
      pixel_cnt_sclr <= '1';
      if (bit_cnt = 7) AND (line_cnt = max_line) then
        ss_ns <= init;
        bit_cnt_sclr <= '1';
        line_cnt_sclr <= '1';
      elsif (bit_cnt = 7) then
        ss_ns <= dis_a;
        bit_cnt_sclr <= '1';
        line_cnt_en <= '1';
      else
        ss_ns <= dis_a;
        bit_cnt_en <= '1';
      end if;
    end case;
--
-- generate output bits from memory as multiplexer or for setting registers
--
  if (reg_set = '1') then out_ff_d(12 downto 7) <= "111111"; --all outputs set to one, used to write to registers
  elsif (reg_clr = '1') then out_ff_d(12 downto 7) <= "000000"; --all outputs set to zero, bit sequence determined in state machine
  else
    case bit_cnt is
      when "000" =>
        out_ff_d(7)  <= mem1(0); out_ff_d(9)  <= mem1(8);  out_ff_d(11) <= mem1(16);--rgb memory 1
        out_ff_d(8)  <= mem2(0); out_ff_d(10) <= mem2(8);  out_ff_d(12) <= mem2(16);--rgb memory 2
      when "001" =>
        out_ff_d(7)  <= mem1(1); out_ff_d(9)  <= mem1(9);  out_ff_d(11) <= mem1(17);
        out_ff_d(8)  <= mem2(1); out_ff_d(10) <= mem2(9);  out_ff_d(12) <= mem2(17);
      when "010" =>
        out_ff_d(7)  <= mem1(2); out_ff_d(9)  <= mem1(10); out_ff_d(11) <= mem1(18);
        out_ff_d(8)  <= mem2(2); out_ff_d(10) <= mem2(10); out_ff_d(12) <= mem2(18);
      when "011" =>
        out_ff_d(7)  <= mem1(3); out_ff_d(9)  <= mem1(11); out_ff_d(11) <= mem1(19);
        out_ff_d(8)  <= mem2(3); out_ff_d(10) <= mem2(11); out_ff_d(12) <= mem2(19);
      when "100" =>
        out_ff_d(7)  <= mem1(4); out_ff_d(9)  <= mem1(12); out_ff_d(11) <= mem1(20);
        out_ff_d(8)  <= mem2(4); out_ff_d(10) <= mem2(12); out_ff_d(12) <= mem2(20);
      when "101" =>
        out_ff_d(7)  <= mem1(5); out_ff_d(9)  <= mem1(13); out_ff_d(11) <= mem1(21);
        out_ff_d(8)  <= mem2(5); out_ff_d(10) <= mem2(13); out_ff_d(12) <= mem2(21);
      when "110" =>
        out_ff_d(7)  <= mem1(6); out_ff_d(9)  <= mem1(14); out_ff_d(11) <= mem1(22);
        out_ff_d(8)  <= mem2(6); out_ff_d(10) <= mem2(14); out_ff_d(12) <= mem2(22);
      when "111" =>
        out_ff_d(7)  <= mem1(7); out_ff_d(9)  <= mem1(15); out_ff_d(11) <= mem1(23);
        out_ff_d(8)  <= mem2(7); out_ff_d(10) <= mem2(15); out_ff_d(12) <= mem2(23);
    end case;
  end if;
--
  end process comb_proc;  
end;
